#' evaluateMissingCandidateSolutions
#'
#' \code{evaluateMissingCandidateSolutions} evaluates all non-evaluated
#' candidate solutions in a gived data.frame. This function is used as a convenience function
#' for codes that are automatically generated by the SPOT-GUI.
#'
#' @param currX A matrix containing all candidate solutions. One candidate per row.
#' @param currY A column vector with all known objective function results for the given matrix of
#' candidate solutions. Default = NULL (In this case all candidate solutions will be evaluated).
#' Missing values have to be marked as NA.
#' @param fun The objective function on which the given candidate solutions shall be evaluated.
#' @return y An updated column vector with evaluation results for all candidate soltuions
#' given in currX
#' @examples
#' library(SPOT)
#' spotData <- NULL
#' #Generating DOE
#' spotData$x <- designLHD(x = NULL, lower = c(-5, -5), upper = c(5, 5),
#'                         control = list(size = 10,
#'                              types = c("numeric", "numeric")))
#'
#' #Evaluating Candidate Solutions
#' spotData$y <- evaluateMissingCandidateSolutions(
#'         currX = spotData$x, currY = spotData$y, fun = funSphere)
#'
#' #Build model on evaluated data
#' spotData$modelFit <- buildKriging(as.matrix(spotData$x),as.matrix(spotData$y))
#'
#' @export
evaluateMissingCandidateSolutions <- function(currX, currY = NULL, fun){
    y <- currY
    if(length(dim(currX))<2){
        currX <- matrix(currX,ncol=1)
    }
    if(is.null(y)){
        results <- fun(currX)
        y <- results
    }else if(any(is.na(y))){
        ind <- which(is.na(y))
        results <- fun(currX[ind,,drop=F])
        y[ind] <- results
    }else if(nrow(currX) > length(y)){
        results <- fun(currX[(length(y)+1):nrow(currX),,drop=F])
        y <- c(y,results)
    }
    return(y)
}
