% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/logistic2ph.R
\name{logistic2ph}
\alias{logistic2ph}
\title{Sieve maximum likelihood estimator (SMLE) for two-phase logistic regression problems}
\usage{
logistic2ph(
  Y_unval = NULL,
  Y = NULL,
  X_unval = NULL,
  X = NULL,
  Z = NULL,
  Bspline = NULL,
  data = NULL,
  hn_scale = 1,
  noSE = FALSE,
  TOL = 1e-04,
  MAX_ITER = 1000,
  verbose = FALSE
)
}
\arguments{
\item{Y_unval}{Column name of the error-prone or unvalidated binary outcome. This argument is required.}

\item{Y}{Column name that stores the validated value of \code{Y_unval} in the second phase. Subjects with missing values of \code{Y} are considered as those not selected in the second phase. This argument is required.}

\item{X_unval}{Specifies the columns of the error-prone covariates. This argument is required.}

\item{X}{Specifies the columns that store the validated values of \code{X_unval} in the second phase. Subjects with missing values of \code{X} are considered as those not selected in the second phase. This argument is required.}

\item{Z}{Specifies the columns of the accurately measured covariates. This argument is optional.}

\item{Bspline}{Specifies the columns of the B-spline basis. This argument is required.}

\item{data}{Specifies the name of the dataset. This argument is required.}

\item{hn_scale}{Specifies the scale of the perturbation constant in the variance estimation. For example, if \code{hn_scale = 0.5}, then the perturbation constant is \eqn{0.5n^{-1/2}}, where \eqn{n} is the first-phase sample size. The default value is \code{1}. This argument is optional.}

\item{noSE}{If \code{TRUE}, then the variances of the parameter estimators will not be estimated. The default value is \code{FALSE}. This argument is optional.}

\item{TOL}{Specifies the convergence criterion in the EM algorithm. The default value is \code{1E-4}. This argument is optional.}

\item{MAX_ITER}{Maximum number of iterations in the EM algorithm. The default number is \code{1000}. This argument is optional.}

\item{verbose}{If \code{TRUE}, then show details of the analysis. The default value is \code{FALSE}.}
}
\value{
\item{coefficients}{Stores the analysis results.}
\item{outcome_err_coefficients}{Stores the outcome error model results.}
\item{Bspline_coefficients}{Stores the final B-spline coefficient estimates.}
\item{covariance}{Stores the covariance matrix of the regression coefficient estimates.}
\item{converge}{In parameter estimation, if the EM algorithm converges, then \code{converge = TRUE}. Otherwise, \code{converge = FALSE}.}
\item{converge_cov}{In variance estimation, if the EM algorithm converges, then \code{converge_cov = TRUE}. Otherwise, \code{converge_cov = FALSE}.}
\item{converge_msg}{In parameter estimation, if the EM algorithm does not converge, then \code{converged_msg} is a string description.}
}
\description{
This function returns the sieve maximum likelihood estimators (SMLE) for the logistic regression model from Lotspeich et al. (2021).
}
\examples{
 set.seed(918)

 # Set sample sizes ----------------------------------------
 N <- 1000 # Phase-I = N
 n <- 250 # Phase-II/audit size = n

 # Generate true values Y, Xb, Xa --------------------------
 Xa <- rbinom(n = N, size = 1, prob = 0.25)
 Xb <- rbinom(n = N, size = 1, prob = 0.5)
 Y <- rbinom(n = N, size = 1,prob = (1 + exp(-(- 0.65 - 0.2 * Xb - 0.1 * Xa))) ^ (- 1))

 # Generate error-prone Xb* from error model P(Xb*|Xb,Xa) --
 sensX <- specX <- 0.75
 delta0 <- - log(specX / (1 - specX))
 delta1 <- - delta0 - log((1 - sensX) / sensX)
 Xbstar <- rbinom(n = N, size = 1,
                  prob = (1 + exp(- (delta0 + delta1 * Xb + 0.5 * Xa))) ^ (- 1))

 # Generate error-prone Y* from error model P(Y*|Xb*,Y,Xb,Xa)
 sensY <- 0.95
 specY <- 0.90
 theta0 <- - log(specY / (1 - specY))
 theta1 <- - theta0 - log((1 - sensY) / sensY)
 Ystar <- rbinom(n = N, size = 1,
   prob = (1 + exp(- (theta0 - 0.2 * Xbstar + theta1 * Y - 0.2 * Xb - 0.1 * Xa))) ^ (- 1))

 ## V is a TRUE/FALSE vector where TRUE = validated --------
 V <- seq(1, N) \%in\% sample(x = seq(1, N), size = n, replace = FALSE)

 # Build dataset --------------------------------------------
 sdat <- cbind(id = 1:N, Y, Xb, Ystar, Xbstar, Xa)
 # Make Phase-II variables Y, Xb NA for unaudited subjects ---
 sdat[!V, c("Y", "Xb")] <- NA

 # Fit model -----------------------------------------------
 ### Construct B-spline basis -------------------------------
 ### Since Xb* and Xa are both binary, reduces to indicators --
 nsieve <- 4
 B <- matrix(0, nrow = N, ncol = nsieve)
 B[which(Xa == 0 & Xbstar == 0), 1] <- 1
 B[which(Xa == 0 & Xbstar == 1), 2] <- 1
 B[which(Xa == 1 & Xbstar == 0), 3] <- 1
 B[which(Xa == 1 & Xbstar == 1), 4] <- 1
 colnames(B) <- paste0("bs", seq(1, nsieve))
 sdat <- cbind(sdat, B)
 smle <- logistic2ph(Y_unval = "Ystar",
   Y = "Y",
   X_unval = "Xbstar",
   X = "Xb",
   Z = "Xa",
   Bspline = colnames(B),
   data = sdat,
   noSE = FALSE,
   MAX_ITER = 1000,
   TOL = 1E-4)
}
\references{
Lotspeich, S. C., Shepherd, B. E., Amorim, G. G. C., Shaw, P. A., & Tao, R. (2021). Efficient odds ratio estimation under two-phase sampling using error-prone data from a multi-national HIV research cohort. \emph{Biometrics, biom.13512.} https://doi.org/10.1111/biom.13512
}
