% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Baseline.R
\name{plotBaselineSummary}
\alias{plotBaselineSummary}
\title{Plots BASELINe summary statistics}
\usage{
plotBaselineSummary(baseline, idColumn, groupColumn = NULL,
  groupColors = NULL, subsetRegions = NULL, facetBy = c("region",
  "group"), style = c("mean"), size = 1, silent = FALSE, ...)
}
\arguments{
\item{baseline}{either a data.frame returned from \link{summarizeBaseline}
or a \code{Baseline} object returned from \link{groupBaseline}
containing selection probability density functions and summary 
statistics.}

\item{idColumn}{name of the column in \code{baseline} containing primary identifiers. 
If the input is a \code{Baseline} object, then this will be a column
in the \code{stats} slot of \code{baseline}.}

\item{groupColumn}{name of the column in \code{baseline} containing secondary grouping 
identifiers. If the input is a \code{Baseline} object, then this will 
be a column in the \code{stats} slot of \code{baseline}.}

\item{groupColors}{named vector of colors for entries in \code{groupColumn}, with 
names defining unique values in the \code{groupColumn} and values
being colors. Also controls the order in which groups appear on the
plot. If \code{NULL} alphabetical ordering and a default color palette 
will be used. Has no effect if \code{facetBy="group"}.}

\item{subsetRegions}{character vector defining a subset of regions to plot, correspoding 
to the regions for which the \code{baseline} data was calculated. If
\code{NULL} all regions in \code{baseline} are plotted.}

\item{facetBy}{one of c("group", "region") specifying which category to facet the
plot by, either values in \code{groupColumn} ("group") or regions
defined in \code{baseline} ("region"). The data that is not used
for faceting will be color coded.}

\item{style}{type of plot to draw. One of:
\itemize{
  \item \code{"mean"}:     plots the mean and confidence interval for
                           the selection scores of each value in 
                           \code{idColumn}. Faceting and coloring
                           are determine by values in \code{groupColumn}
                           and regions defined in \code{baseline}, 
                           depending upon the \code{facetBy} argument.
}}

\item{size}{numeric scaling factor for lines, points and text in the plot.}

\item{silent}{if \code{TRUE} do not draw the plot and just return the ggplot2 
object; if \code{FALSE} draw the plot.}

\item{...}{additional arguments to pass to ggplot2::theme.}
}
\value{
A ggplot object defining the plot.
}
\description{
\code{plotBaselineSummary} plots a summary of the results of selection analysis 
using the BASELINe method.
}
\examples{
# Subset example data
db <- subset(InfluenzaDb, CPRIMER \%in\% c("IGHA","IGHM") & 
                          BARCODE \%in\% c("RL016","RL018","RL019","RL021"))

# Calculate BASELINe
# By default, calcBaseline collapses the sequences in the db by the column "CLONE",
# calculates the numbers of observed mutations and expected frequencies of mutations,
# as defined in the IMGT_V_NO_CDR3 and using the HS5FModel targeting model.
# Then, it calculates  the BASELINe posterior probability density functions (PDFs) for
# sequences in the updated db files; using the focused test statistic
db_baseline <- calcBaseline(db, 
                            sequenceColumn="SEQUENCE_IMGT",
                            germlineColumn="GERMLINE_IMGT_D_MASK", 
                            testStatistic="focused",
                            regionDefinition=IMGT_V_NO_CDR3,
                            targetingModel = HS5FModel,
                            nproc = 1)
 
# Grouping the PDFs by the BARCODE and CPRIMER columns in the db, corresponding 
# respectively to sample barcodes and the constant region isotype primers.
baseline <- groupBaseline(db_baseline, groupBy=c("BARCODE", "CPRIMER"))

# Plot mean and confidence interval
plotBaselineSummary(baseline, "BARCODE", "CPRIMER", style="mean")
plotBaselineSummary(baseline, "BARCODE", "CPRIMER", subsetRegions="CDR", style="mean")
plotBaselineSummary(baseline, "BARCODE", "CPRIMER", facetBy="group", style="mean")

# Reorder and recolor groups
group_colors <- c("IGHM"="darkorchid", "IGHA"="steelblue")
plotBaselineSummary(baseline, "BARCODE", "CPRIMER", groupColors=group_colors, style="mean")

# Plot subset of data
stats <- subset(getBaselineStats(baseline), BARCODE \%in\% c("RL018", "RL019"))
plotBaselineSummary(stats, "BARCODE", "CPRIMER", groupColors=group_colors, style="mean")

}
\seealso{
Takes as input either a \link{Baseline} object returned by \link{groupBaseline} 
          or a data.frame returned from \link{summarizeBaseline}.

Other selection analysis functions: \code{\link{calcBaseline}},
  \code{\link{groupBaseline}},
  \code{\link{plotBaselineDensity}},
  \code{\link{summarizeBaseline}}
}

