% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/igraph_object.R
\name{Graph}
\alias{Graph}
\title{Graph visualisation}
\usage{
Graph(
  adjacency,
  node_label = NULL,
  node_colour = NULL,
  node_shape = NULL,
  edge_colour = "grey60",
  label_colour = "grey20",
  mode = "undirected",
  weighted = FALSE,
  satellites = FALSE
)
}
\arguments{
\item{adjacency}{adjacency matrix or output of \code{\link{GraphicalModel}}.}

\item{node_label}{optional vector of node labels. This vector must contain as
many entries as there are rows/columns in the adjacency matrix and must be
in the same order (the order is used to assign labels to nodes).}

\item{node_colour}{optional vector of node colours. This vector must contain
as many entries as there are rows/columns in the adjacency matrix and must
be in the same order (the order is used to assign colours to nodes).
Integers, named colours or RGB values can be used.}

\item{node_shape}{optional vector of node shapes. This vector must contain as
many entries as there are rows/columns in the adjacency matrix and must be
in the same order (the order is used to assign shapes to nodes). Possible
values are \code{"circle"}, \code{"square"}, \code{"triangle"} or
\code{"star"}.}

\item{edge_colour}{optional character string for edge colour. Integers, named
colours or RGB values can be used.}

\item{label_colour}{optional character string for label colour. Integers,
named colours or RGB values can be used.}

\item{mode}{character string indicating how the adjacency matrix should be
interpreted. Possible values include \code{"undirected"} or
\code{"directed"} (see \code{\link[igraph]{graph_from_adjacency_matrix}}).}

\item{weighted}{indicating if entries of the adjacency matrix should define
edge width. If \code{weighted=FALSE}, an unweighted igraph object is
created, all edges have the same width. If \code{weighted=TRUE}, edge width
is defined by the corresponding value in the adjacency matrix. If
\code{weighted=NULL}, nodes are linked by as many edges as indicated in the
adjacency matrix (integer values are needed).}

\item{satellites}{logical indicating if unconnected nodes (satellites) should
be included in the igraph object.}
}
\value{
An igraph object.
}
\description{
Produces an \code{\link[igraph:igraph-package]{igraph}} object from an
adjacency matrix.
}
\details{
All functionalities implemented in
  \code{\link[igraph:igraph-package]{igraph}} can be used on the output.
  These include cosmetic changes for the visualisation, but also various
  tools for network analysis (including topological properties and community
  detection).

  The R package \code{\link[visNetwork:visDocumentation]{visNetwork}} offers
  interactive network visualisation tools. An
  \code{\link[igraph:igraph-package]{igraph}} object can easily be converted
  to a \code{\link[visNetwork:visDocumentation]{visNetwork}} object (see
  example below).

  For Cytoscape users, the \code{\link[RCy3:RCy3]{RCy3}} package can be used
  to open the network in Cytoscape.
}
\examples{
\donttest{
## From adjacency matrix

# Un-weighted
adjacency <- SimulateAdjacency(pk = 20, topology = "scale-free")
plot(Graph(adjacency))

# Weighted
adjacency <- adjacency * runif(prod(dim(adjacency)))
adjacency <- adjacency + t(adjacency)
plot(Graph(adjacency, weighted = TRUE))

# Node colours and shapes
plot(Graph(adjacency, weighted = TRUE, node_shape = "star", node_colour = "red"))


## From stability selection outputs

# Graphical model
set.seed(1)
simul <- SimulateGraphical(pk = 20)
stab <- GraphicalModel(xdata = simul$data)
plot(Graph(stab))

# Sparse PLS
set.seed(1)
simul <- SimulateRegression(n = 50, pk = c(5, 5, 5), family = "gaussian")
x <- simul$xdata
y <- simul$ydata
stab <- BiSelection(
  xdata = simul$xdata, ydata = simul$ydata,
  family = "gaussian", ncomp = 3,
  LambdaX = 1:(ncol(x) - 1),
  implementation = SparsePLS
)
plot(Graph(stab))


## Tools from other packages

# Applying some igraph functionalities
adjacency <- SimulateAdjacency(pk = 20, topology = "scale-free")
mygraph <- Graph(adjacency)
igraph::degree(mygraph)
igraph::betweenness(mygraph)
igraph::shortest_paths(mygraph, from = 1, to = 2)
igraph::walktrap.community(mygraph)

# Interactive view using visNetwork
if (requireNamespace("visNetwork", quietly = TRUE)) {
  vgraph <- mygraph
  igraph::V(vgraph)$shape <- rep("dot", length(igraph::V(vgraph)))
  v <- visNetwork::visIgraph(vgraph)
  mylayout <- as.matrix(v$x$nodes[, c("x", "y")])
  mylayout[, 2] <- -mylayout[, 2]
  plot(mygraph, layout = mylayout)
}

# Opening in Cytoscape using RCy3
if (requireNamespace("RCy3", quietly = TRUE)) {
  # Make sure that Cytoscape is open before running the following line
  # RCy3::createNetworkFromIgraph(mygraph)
}
}

}
\seealso{
\code{\link{Adjacency}}, \code{\link{GraphicalModel}},
  \href{https://igraph.org/r/}{igraph manual},
  \href{http://datastorm-open.github.io/visNetwork/}{visNetwork manual},
  \href{https://cytoscape.org}{Cytoscape}
}
