% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/carbmodel.r
\name{carbmodel}
\alias{carbmodel}
\title{Function that produces \eqn{\delta^{18}O}{δ18O} and \eqn{\Delta_{47}}{Δ47} records}
\usage{
carbmodel(
  time,
  SST,
  GR,
  d18Ow,
  D,
  d18O_fun = "KimONeil97",
  D47_fun = "Bernasconi18",
  AV = FALSE,
  plot = FALSE
)
}
\arguments{
\item{time}{Time vector (values in years)}

\item{SST}{A vector containing temperature data (values in degrees C; length
must be equal to that of \code{time})}

\item{GR}{Growth rate vector (values in same time unit as \code{time} (years);
length must be equal to that of \code{time})}

\item{d18Ow}{A vector containing data on the \eqn{\delta^{18}O}{δ18O} value of
the precipitation fluid (values in permille VSMOW; length must be equal to
that of \code{time})}

\item{D}{Depth vector (values in same depth unit as \code{GR})}

\item{d18O_fun}{String containing the name of the transfer function used to
convert temperature and \eqn{\delta^{18}O_{w}}{δ18Ow} to
\eqn{\delta^{18}O_{c}}{δ18Oc} data (for example: \code{"KimONeil97"} or
\code{"GrossmanKu86"}). Defaults to Kim and O'Neil (1997).}

\item{D47_fun}{String containing the name of the transfer function used to
convert temperature to \eqn{\Delta_{47}}{Δ47} data (for example:
\code{"Bernasconi18"} or \code{"Jautzy20"}). Defaults to Bernasconi et al.,
2018).}

\item{AV}{Should the subsampling take into account the mean value within the
sample interval? \code{TRUE/FALSE} If \code{FALSE}, the interpolated value
corresponding to the exact position is used instead of the mean of the interval}

\item{plot}{Should the result be plotted? \code{TRUE/FALSE}}
}
\value{
A matrix containing subsampled time, depth,
\eqn{\delta^{18}O_{c}}{δ18Oc} and \eqn{\Delta_{47}}{Δ47} values:
\code{"Tnew"}): New time vector after subsampling
\code{"D"}): New depth vector after subsampling
\code{"d18Oc"}): Vector listing \eqn{\delta^{18}O_{c}}{δ18Oc} values for each
sample
\code{"D47"}): Vector listing \eqn{\Delta_{47}}{Δ47} values for each sample
}
\description{
Takes vectors of time, temperature, growth rate and \eqn{\delta^{18}O}{δ18O}
of the fluid and converts them into a \eqn{\delta^{18}O}{δ18O} and
\eqn{\Delta_{47}}{Δ47} record. The \eqn{\delta^{18}O}{δ18O} and
\eqn{\Delta_{47}}{Δ47} values are calculated for every depth value provided
in the \code{D} vector. By default, the empirical transfer function by Kim
and O'Neil (1997) is used to produce the \eqn{\delta^{18}O}{δ18O} record, but
other transfer functions (e.g. Grossman and Ku, 1986) are also supported. The
default transfer function for converting temperature data to
\eqn{\Delta_{47}}{Δ47} data is based on Bernasconi et al. (2018), but other
transfer functions (e.g. Jautzy et al., 2020) are also supported.
}
\examples{
# Create test data (= ideal case)
# Set boundary conditions
Td <- seq(1, 12 * 365, 1) # Create timeline of 12 years in days
Ty <- Td / 365 # Convert to years
MAT <- 20 # Set mean annual temperature
Amp <- 10 # Set seasonal amplitude
Sext <- 2 * Amp # Calculate extent of seasonal variability
TSD <- 1.5 # Set the degree of random nonâ€“seasonal noise on the SST curve
# ("weather")
SST <- rnorm(length(Ty), MAT + Amp * sin(2 * pi * Ty), TSD) # Create virtual
# daily SST data
GR <- rep(10 / 365, length(Ty)) # Set growth rate to 10 mm/yr and create daily
# GR vector
DSD <- 0.6 # Set the degree of random nonâ€“seasonal noise on the d18Osw curve
# ("salinity fluctuations")
d18Osw<-rnorm(length(Ty), rep(0, length(Ty)), DSD) # Set d18Osw to 0 permille
# VSMOW, create daily d18Osw vector
SR <- 0.75 # Set sampling resolution to 0.75 mm
# Create vector for all samples along entire shell length by applying constant
# sampling resolution
D <- seq(SR, sum(GR), SR)
# Calculate virtual data
newdata <- carbmodel(Ty, SST, GR, d18Osw, D, AV = TRUE)
}
\references{
package dependencies: ggplot2, gridExtra
function dependencies: subsample, subsample_mean
Grossman, E.L., Ku, T., Oxygen and carbon isotope fractionation in biogenic
aragonite: temperature effects, \emph{Chemical Geology} \strong{1986}, \emph{59.1}, 59–74.
\url{https://doi.org/bvpzws}

Kim, S., O'Niel, J.R., Equilibrium and nonequilibrium oxygen
isotope effects in synthetic carbonates, \emph{Geochimica et Cosmochimica Acta}
\strong{1997}, \emph{61.16}, 3461–3475.
\url{https://doi.org/c7bwbp}

Dettman, D.L., Reische, A.K., Lohmann, K.C., Controls on the stable isotope
composition of seasonal growth bands in aragonitic fresh–water bivalves
(Unionidae), \emph{Geochimica et Cosmochimica Acta} \strong{1999}, \emph{63.7–8}, 1049–1057.
\url{https://doi.org/cbb7zc}

Brand, W.A., Coplen, T.B., Vogl, J., Rosner, M., Prohaska, T., Assessment of
international reference materials for isotope–ratio analysis (IUPAC Technical
Report), \emph{Pure and Applied Chemistry} \strong{2014}, \emph{86.3}, 425–467.
\url{https://doi.org/fpc2}

Kele, S., Breitenbach, S. F., Capezzuoli, E., Meckler, A. N., Ziegler, M.,
Millan, I. M., Kluge, T., Deák, J., Hanselmann, K. and John, C. M.,
Temperature dependence of oxygen– and clumped isotope fractionation in
carbonates: a study of travertines and tufas in the 6–95 C temperature range,
\emph{Geochimica et Cosmochimica Acta} \strong{2015}, 168, 172–192.
\url{https://doi.org/f7sgn6}

Bernasconi, S.M., Müller, I.A., Bergmann, K.D., Breitenbach, S.F., Fernandez,
A., Hodell, D.A., Jaggi, M., Meckler, A.N., Millan, I. and Ziegler, M.,
Reducing uncertainties in carbonate–clumped isotope analysis through
consistent carbonate based standardization. \emph{Geochemistry, Geophysics,
Geosystems} \strong{2018}, 19–9, 2895–2914.
\url{https://doi.org/gfmjrw}

Petersen, S. V., Defliese, W. F., Saenger, C., Daëron, M., Huntington, K. W.,
John, C. M., Kelson, J. R., Bernasconi, S. M., Colman, A. S., Kluge, T.,
Olack, G. A., Schauer, A. J., Bajnai, D., Bonifacie, M., Breitenbach, S. F.
M., Fiebig, J., Fernandez, A. B., Henkes, G. A., Hodell, D., Katz, A., Kele,
S., Lohmann, K. C., Passey, B. H., Peral, M. Y., Petrizzo, D. A., Rosenheim,
B. E., Tripati, A., Venturelli, R., Young, E. D. and Winkelstern, I. Z.,
Effects of Improved 17O Correction on Interlaboratory Agreement in Clumped
Isotope Calibrations, Estimates of Mineral–Specific Offsets, and Temperature
Dependence of Acid Digestion Fractionation, \emph{Geochemistry, Geophysics,
Geosystems} *\emph{2019}, 20–7, 3495–3519.
\url{https://doi.org/ggrc39}

Jautzy, J. J., Savard, M. M., Dhillon, R. S., Bernasconi, S. M. and Smirnoff,
A., Clumped isotope temperature calibration for calcite: Bridging theory and
experimentation, \emph{Geochemical Perspectives Letters} \strong{2020}, 14, 36–41.
\url{https://doi.org/fpc3}
}
