% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/residuals.R
\name{residuals.sdmTMB}
\alias{residuals.sdmTMB}
\title{Residuals method for sdmTMB models}
\usage{
\method{residuals}{sdmTMB}(
  object,
  type = c("mle-laplace", "mle-mcmc", "mvn-laplace", "response", "pearson"),
  model = c(1, 2),
  mcmc_samples = NULL,
  ...
)
}
\arguments{
\item{object}{An \code{\link[=sdmTMB]{sdmTMB()}} model}

\item{type}{Type of residual. See details.}

\item{model}{Which delta/hurdle model component?}

\item{mcmc_samples}{A vector of MCMC samples of the linear predictor in link
space. See the
\href{https://github.com/pbs-assess/sdmTMBextra}{sdmTMBextra} package.}

\item{...}{Passed to residual function. Only \code{n} works for binomial.}
}
\value{
A vector of residuals.
}
\description{
See the residual-checking vignette: \code{browseVignettes("sdmTMB")} or \href{https://pbs-assess.github.io/sdmTMB/articles/residual-checking.html}{on the documentation site}.
See notes about types of residuals in 'Details' section below.
}
\details{
Types of residuals currently supported:

\strong{\code{"mle-laplace"}} refers to randomized quantile residuals (Dunn &
Smyth 1996), which are also known as probability integral transform (PIT)
residuals (Smith 1985). Under model assumptions, these should be distributed
as standard normal with the following caveat: the Laplace approximation used
for the latent/random effects can cause these residuals to deviate from the
standard normal assumption even if the model is consistent with the data
(Thygesen et al. 2017). Therefore, \strong{these residuals are fast to calculate
but can be unreliable.}

\strong{\code{"mle-mcmc"}} refers to randomized quantile residuals where the fixed
effects are fixed at their MLE (maximum likelihood estimate) values and the
random effects are sampled with MCMC via tmbstan/Stan. As proposed in
Thygesen et al. (2017) and used in Rufener et al. (2021). Under model
assumptions, these should be distributed as standard normal. \strong{These
residuals are theoretically preferred over the regular Laplace approximated
randomized-quantile residuals, but will be considerably slower to
calculate.}

See the \href{https://github.com/pbs-assess/sdmTMBextra}{\pkg{sdmTMBextra}}
package for the function \code{predict_mle_mcmc()},
which can generate the MCMC samples to pass to the \code{mcmc_samples} argument.
Ideally MCMC is run until convergence and then the last iteration can be
used for residuals. MCMC samples are defined by \code{mcmc_iter - mcmc_warmup}.
The Stan model can be printed with \code{print_stan_model = TRUE} to check.
The defaults may not be sufficient for many models.

\strong{\code{"mvn-laplace"}} is the same as \code{"mle-laplace"} except the parameters are
based on simulations drawn from the assumed multivariate normal distribution
(using the joint precision matrix).

\strong{\code{"response"}} refers to response residuals: observed minus predicted.
}
\examples{
if (inla_installed()) {

  mesh <- make_mesh(pcod_2011, c("X", "Y"), cutoff = 10)
  fit <- sdmTMB(
    present ~ as.factor(year) + poly(depth, 3),
    data = pcod_2011, mesh = mesh,
    family = binomial()
  )

  # response residuals will be not be normally distributed unless
  # the family is Gaussian:
  r0 <- residuals(fit, type = "response")
  qqnorm(r0)
  qqline(r0)

  # quick but can have issues because of Laplace approximation:
  r1 <- residuals(fit, type = "mle-laplace")
  qqnorm(r1)
  qqline(r1)

  # see also "mle-mcmc" residuals with the help of the sdmTMBextra package
}
}
\references{
Dunn, P.K. & Smyth, G.K. (1996). Randomized Quantile Residuals. Journal of
Computational and Graphical Statistics, 5, 236–244.

Smith, J.Q. (1985). Diagnostic checks of non-standard time series models.
Journal of Forecasting, 4, 283–291.

Rufener, M.-C., Kristensen, K., Nielsen, J.R., and Bastardie, F. 2021.
Bridging the gap between commercial fisheries and survey data to model the
spatiotemporal dynamics of marine species. Ecological Applications. e02453.
\doi{10.1002/eap.2453}

Thygesen, U.H., Albertsen, C.M., Berg, C.W., Kristensen, K., and Nielsen, A.
2017. Validation of ecological state space models using the Laplace
approximation. Environ Ecol Stat 24(2): 317–339.
\doi{10.1007/s10651-017-0372-4}
}
