% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tmb-sim.R
\name{simulate.sdmTMB}
\alias{simulate.sdmTMB}
\title{Simulate from a fitted sdmTMB model}
\usage{
\method{simulate}{sdmTMB}(
  object,
  nsim = 1L,
  seed = sample.int(1e+06, 1L),
  params = c("mle", "mvn"),
  model = c(NA, 1, 2),
  re_form = NULL,
  tmbstan_model = NULL,
  ...
)
}
\arguments{
\item{object}{sdmTMB model}

\item{nsim}{Number of response lists to simulate. Defaults to 1.}

\item{seed}{Random number seed}

\item{params}{Whether the parameters used in the simulation should come from
the Maximum Likelihood Estimate (\code{"mle"}) or from new draws from the joint
precision matrix assuming they are multivariate normal distributed
(\code{"mvn"}).}

\item{model}{If a delta/hurdle model, which model to simulate from?
\code{NA} = combined, \code{1} = first model, \code{2} = second mdoel.}

\item{re_form}{\code{NULL} to specify a simulation conditional on fitted random
effects (this only simulates observation error). \code{~0} or \code{NA} to simulate
new random affects (smoothers, which internally are random effects, will
not be simulated as new).}

\item{tmbstan_model}{An optional model fit via \code{\link[tmbstan:tmbstan]{tmbstan::tmbstan()}}. If
provided the parameters will be drawn from the MCMC samples and new
observation error will be added. See the example in \code{\link[=extract_mcmc]{extract_mcmc()}}.}

\item{...}{Extra arguments (not used)}
}
\value{
Returns a matrix; number of columns is \code{nsim}.
}
\description{
\code{simulate.sdmTMB} is an S3 method for producing a matrix of simulations from
a fitted model. This is similar to \code{\link[lme4:simulate.merMod]{lme4::simulate.merMod()}} and
\code{\link[glmmTMB:simulate.glmmTMB]{glmmTMB::simulate.glmmTMB()}}. It can be used with the \pkg{DHARMa} package
among other uses.
}
\examples{
if (inla_installed()) {

# start with some data simulated from scratch:
set.seed(1)
predictor_dat <- data.frame(X = runif(300), Y = runif(300), a1 = rnorm(300))
mesh <- make_mesh(predictor_dat, xy_cols = c("X", "Y"), cutoff = 0.1)
dat <- sdmTMB_simulate(
  formula = ~ 1 + a1,
  data = predictor_dat,
  mesh = mesh,
  family = poisson(),
  range = 0.5,
  sigma_O = 0.2,
  seed = 42,
  B = c(0.2, -0.4) # B0 = intercept, B1 = a1 slope
)
fit <- sdmTMB(observed ~ 1 + a1, data = dat, family = poisson(), mesh = mesh)

# simulate from the model:
s1 <- simulate(fit, nsim = 300)
dim(s1)

# test whether fitted models are consistent with the observed number of zeros:
sum(s1 == 0)/length(s1)
sum(dat$observed == 0) / length(dat$observed)

# use the residuals with DHARMa:
if (require("DHARMa", quietly = TRUE)) {
  pred_fixed <- fit$family$linkinv(predict(fit)$est_non_rf)
  r <- DHARMa::createDHARMa(
    simulatedResponse = s1,
    observedResponse = dat$observed,
    fittedPredictedResponse = pred_fixed
  )
  plot(r)
  DHARMa::testResiduals(r)
  DHARMa::testSpatialAutocorrelation(r, x = dat$X, y = dat$Y)
  DHARMa::testZeroInflation(r)
}

# simulate with the parameters drawn from the joint precision matrix:
s2 <- simulate(fit, nsim = 1, params = "MVN")

# simulate with new random fields:
s3 <- simulate(fit, nsim = 1, re_form = ~ 0)

# simulate with new random fields and new parameter draws:
s3 <- simulate(fit, nsim = 1, params = "MVN", re_form = ~ 0)

# simulate from a Stan model fit with new observation error:
\donttest{
if (require("tmbstan", quietly = TRUE)) {
  stan_fit <- tmbstan::tmbstan(fit$tmb_obj, iter = 110, warmup = 100, chains = 1)
  # make sure `nsim` is <= number of samples from rstan
  s3 <- simulate(fit, nsim = 10, tmbstan_model = stan_fit)
}
}
}
}
\seealso{
\code{\link[=sdmTMB_simulate]{sdmTMB_simulate()}}, \code{\link[=dharma_residuals]{dharma_residuals()}}
}
