% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/et.R
\name{etRep}
\alias{etRep}
\alias{rep.rxEt}
\title{Repeat an rxode2 event table}
\usage{
etRep(
  x,
  times = 1,
  length.out = NA,
  each = NA,
  n = NULL,
  wait = 0,
  id = integer(0),
  samples = c("clear", "use"),
  waitII = c("smart", "+ii"),
  ii = 24
)

\method{rep}{rxEt}(x, ...)
}
\arguments{
\item{x}{An rxode2 event table}

\item{times}{Number of times to repeat the event table}

\item{length.out}{Invalid with rxode2 event tables, will throw an
error if used.}

\item{each}{Invalid with rxode2 event tables, will throw an error
if used.}

\item{n}{The number of times to repeat the event table.  Overrides
\code{times}.}

\item{wait}{Waiting time between each repeated event table.  By
default there is no waiting, or wait=0}

\item{id}{A integer vector of IDs to add or remove from the event
table.  If the event table is identical for each ID, then you
may expand it to include all the IDs in this vector.  All the
negative IDs in this vector will be removed.}

\item{samples}{How to handle samples when repeating an event
table.  The options are:
\itemize{
\item \code{"clear"} Clear sampling records before combining the datasets
\item \code{"use"} Use the sampling records when combining the datasets
}}

\item{waitII}{This determines how waiting times between events are
handled. The options are:
\itemize{
\item \code{"smart"} This "smart" handling of waiting times is the
default option.  In this case, if the waiting time is above the
last observed inter-dose interval in the first combined event
table, then the actual time between doses is given by the wait
time.  If it is smaller than the last observed inter-dose
interval, the time between event tables is given by the inter-dose
interval + the waiting time between event tables.
\item \code{"+ii"} In this case, the wait time is added to the
inter-dose interval no matter the length of the wait time or
inter-dose interval
}}

\item{ii}{When specifying a dose, this is the inter-dose interval
for \code{ss}, \code{addl} and \code{until} options (described below).}

\item{...}{Times or event tables.  They can also be one of the named arguments below.}
}
\value{
An event table
}
\description{
Repeat an rxode2 event table
}
\examples{

\donttest{

library(units)

## These are making the more complex regimens of the rxode2 tutorial

## bid for 5 days
bid <- et(timeUnits="hr") \%>\%
       et(amt=10000,ii=12,until=set_units(5, "days"))

## qd for 5 days
qd <- et(timeUnits="hr") \%>\%
      et(amt=20000,ii=24,until=set_units(5, "days"))

## bid for 5 days followed by qd for 5 days

et <- seq(bid,qd) \%>\% et(seq(0,11*24,length.out=100));


## Now Infusion for 5 days followed by oral for 5 days

##  note you can dose to a named compartment instead of using the compartment number
infusion <- et(timeUnits = "hr") \%>\%
      et(amt=10000, rate=5000, ii=24, until=set_units(5, "days"), cmt="centr")


qd <- et(timeUnits = "hr") \%>\% et(amt=10000, ii=24, until=set_units(5, "days"), cmt="depot")

et <- seq(infusion,qd)


## 2wk-on, 1wk-off

qd <- et(timeUnits = "hr") \%>\% et(amt=10000, ii=24, until=set_units(2, "weeks"), cmt="depot")

et <- seq(qd, set_units(1,"weeks"), qd) \%>\%
     add.sampling(set_units(seq(0, 5.5,by=0.005),weeks))


## You can also repeat the cycle easily with the rep function

qd <-et(timeUnits = "hr") \%>\% et(amt=10000, ii=24, until=set_units(2, "weeks"), cmt="depot")

et <- etRep(qd, times=4, wait=set_units(1,"weeks")) \%>\%
     add.sampling(set_units(seq(0, 12.5,by=0.005),weeks))


}
}
\references{
Wang W, Hallow K, James D (2015). "A Tutorial on rxode2: Simulating
Differential Equation Pharmacometric Models in R." CPT:
Pharmacometrics and Systems Pharmacology, 5(1), 3-10. ISSN 2163-8306,
<URL: https://www.ncbi.nlm.nih.gov/pmc/articles/PMC4728294/>.
}
\seealso{
\code{\link{eventTable}}, \code{\link{add.sampling}},
\code{\link{add.dosing}}, \code{\link{et}},
\code{\link{etRep}}, \code{\link{etRbind}}
}
\author{
Matthew L Fidler, Wenping Wang
}
