% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pkg_ref_cache.R
\name{pkg_ref_cache}
\alias{pkg_ref_cache}
\title{S3 generic to calculate a `pkg_ref` field}
\usage{
pkg_ref_cache(x, name, ..., .class = as.character(name))
}
\arguments{
\item{x}{a package reference object}

\item{name}{the name of the field that needs to be cached}

\item{...}{additional arguments used for computing cached values}

\item{.class}{a class name to use for S3 dispatch, defaulting to the name as
a character value}
}
\value{
a \code{pkg_ref} field

a value to assign to the new field in the package reference object
  environment
}
\description{
Reactively retrieve and cache `pkg_ref` metadata
}
\section{Caching Details}{

\subsection{\code{pkg_ref} class fields}{
  The \code{pkg_ref} class structures an environment with special handling
  for indexing into the \code{pkg_ref} class using the \code{$} or \code{[[}
  operators. For all intents and purposes, the \code{pkg_ref} class is works
  conceptually similar to a lazy, immutable \code{list}, and uses the
  \code{pkg_ref_cache} function internally to lazily retrieve package
  reference fields.
}
\subsection{Lazy metadata caching}{
  Laziness in a \code{pkg_ref} object refers to the delayed evaluation of the
  contents of its fields. Since some metadata is time or computationally
  intensive to retrieve, and unnessary for some assessments, we want to avoid
  that retrieval until it is needed.

  The first time that a field is accessed within a \code{pkg_ref} object
  \code{x}, a corresponding \code{pkg_ref_cache} S3 generic is called. For
  example, when \code{x$description} is first accessed, the \code{pkg_ref}
  object uses the function \code{pkg_ref_cache.description} to attempt to
  retrieve the contents of the corresponding \code{DESCRIPTION} file.

  Often, the way that this data is collected might be different depending on
  the subclass of the \code{pkg_ref}. In the case of the \code{description}
  metadata, a reference to a local install might be able to read in a local
  file directly, whereas a reference to a remote source of metadata might
  require first downloading the file. For this reason, many
  \code{pkg_ref_cache.*} functions are themselves S3 generics that dispatch
  on the class of the \code{pkg_ref} object, allowing for divergent behaviors
  for different source of package metadata.
}
\subsection{\code{pkg_ref} field immutability}{
  Once a field has been calculated, its value is immutable. This behavior was
  chosen because of the long time frame over which package metadata changes,
  rendering it unnecessary to continually reevaluate fields each time they
  are accesssed.

  This means that within an assessment, a given field for a package will only
  ever be calculated once and preserved for downstream use.
}
}

\examples{
# implementing a new field called "first_letter" that is consistently derived
# across all pkg_ref objects:

  pkg_ref_cache.first_letter <- function(x, name, ...) {
    substring(x$name, 1, 1)
  }

  x <- pkg_ref("riskmetric")
  x$first_letter



# implementing a new field called "subclass_enum" that dispatches on
# the subclass of the pkg_ref object:

  pkg_ref_cache.subclass_enum <- function(x, name, ...) {
    UseMethod("pkg_ref_cache.subclass_enum")
  }

  pkg_ref_cache.subclass_enum.pkg_ref <- function(x, name, ...) {
    0
  }

  pkg_ref_cache.subclass_enum.pkg_install <- function(x, name, ...) {
    1
  }

  x$subclass_enum

}
\seealso{
Other package reference cache: 
\code{\link{pkg_ref_cache.archive_release_dates}()},
\code{\link{pkg_ref_cache.bug_reports_host}()},
\code{\link{pkg_ref_cache.bug_reports_url}()},
\code{\link{pkg_ref_cache.bug_reports}()},
\code{\link{pkg_ref_cache.covr_coverage}()},
\code{\link{pkg_ref_cache.description}()},
\code{\link{pkg_ref_cache.downloads}()},
\code{\link{pkg_ref_cache.expression_coverage}()},
\code{\link{pkg_ref_cache.help_aliases}()},
\code{\link{pkg_ref_cache.help}()},
\code{\link{pkg_ref_cache.license}()},
\code{\link{pkg_ref_cache.maintainer}()},
\code{\link{pkg_ref_cache.news_urls}()},
\code{\link{pkg_ref_cache.news}()},
\code{\link{pkg_ref_cache.r_cmd_check}()},
\code{\link{pkg_ref_cache.release_date}()},
\code{\link{pkg_ref_cache.remote_checks}()},
\code{\link{pkg_ref_cache.repo_base_url}()},
\code{\link{pkg_ref_cache.source_control_url}()},
\code{\link{pkg_ref_cache.tarball_url}()},
\code{\link{pkg_ref_cache.vignettes}()},
\code{\link{pkg_ref_cache.web_html}()},
\code{\link{pkg_ref_cache.web_url}()},
\code{\link{pkg_ref_cache.website_urls}()}
}
\concept{package reference cache}
\keyword{internal}
