#' Delete Credentials
#'
#' Delete all the credentials according to a specific user. The credentials
#' (Google Earth Engine, Google Drive and Google Cloud Storage) are created
#' after running successfully `ee_Initialize(...)`. They are saved in
#' the path `rgee::ee_get_earthengine_path()`.
#'
#' @param email Character. Earth Engine user (e.g. `data.colec.fbf`).
#' @param quiet Logical. Suppress info messages.
#' @return No return value, called for cleaning Google Drive, Google Cloud Storage,
#' and/or Earth Engine credentials.
#' @family ee_clean functions
#' @export
ee_clean_credentials <- function(email='not_defined', quiet = FALSE) {
  oauth_func_path <- system.file("python/ee_utils.py", package = "rgee")
  utils_py <- ee_source_python(oauth_func_path)
  ee_path <- ee_utils_py_to_r(utils_py$ee_path())
  email_clean <- gsub("@gmail.com", "", email)

  if (email == 'not_defined') {
    email_clean <- 'ndef'
  }

  path_to_delete <- sprintf("%s/%s", ee_path, email_clean)

  if (!dir.exists(path_to_delete)) {
    if (!quiet) {
      cat('The path:', path_to_delete, ' does not exist!\n')
    }
  }

  if (!quiet && dir.exists(path_to_delete)) {
    cat(
      sprintf("Credentials in %s has been deleted.\n",
              sprintf("%s/%s", ee_path, email_clean)))
  }

  unlink(x = path_to_delete, recursive = TRUE, force = TRUE)
  unlink(list.files(ee_path, "@gmail.com", full.names = TRUE))
  unlink(list.files(ee_path, ".json", full.names = TRUE))
  unlink(list.files(ee_path, "credentials", full.names = TRUE))
  unlink(list.files(ee_path, "rgee_sessioninfo.txt", full.names = TRUE))

  invisible(TRUE)
}

#' Remove rgee system variables from .Renviron
#'
#' @family ee_clean functions
#' @return No return value, called for cleaning environmental variables in their system.
#' @export
ee_clean_pyenv <- function() {
  # Read line by line .Renviron
  home <- Sys.getenv("HOME")
  renv <- file.path(home, ".Renviron")
  if (file.exists(renv)) {
    # Backup original .Renviron before doing anything else here.
    file.copy(renv, file.path(home, ".Renviron_backup"), overwrite = TRUE)

    con <- file(renv, open = "r")
    lines <- as.character()
    ii <- 1

    while (TRUE) {
      line <- readLines(con, n = 1, warn = FALSE)
      if (length(line) == 0) {
        break()
      }
      lines[ii] <- line
      ii <- ii + 1
    }
    close(con)

    # Remove system variables EARTHENGINE_PYTHON
    system_vars <- lines[!grepl("EARTHENGINE_PYTHON|EARTHENGINE_ENV", lines)]
    fileConn <- file(renv)
    writeLines(system_vars, fileConn)
    close(fileConn)
  }
  invisible(TRUE)
}


#' Remove EARTHENGINE_PYTHON_INIT_MESSAGE
#' @noRd
ee_clean_message <- function() {
  # Read line by line .Renviron
  home <- Sys.getenv("HOME")
  renv <- file.path(home, ".Renviron")
  if (file.exists(renv)) {
    # Backup original .Renviron before doing anything else here.
    file.copy(renv, file.path(home, ".Renviron_backup"), overwrite = TRUE)

    con <- file(renv, open = "r")
    lines <- as.character()
    ii <- 1

    while (TRUE) {
      line <- readLines(con, n = 1, warn = FALSE)
      if (length(line) == 0) {
        break()
      }
      lines[ii] <- line
      ii <- ii + 1
    }
    close(con)

    # Remove system variables EARTHENGINE_PYTHON
    system_vars <- lines[!grepl("EARTHENGINE_INIT_MESSAGE", lines)]
    fileConn <- file(renv)
    writeLines(system_vars, fileConn)
    close(fileConn)
  }
  invisible(TRUE)
}

#' Delete files from a either Folder or Bucket
#'
#' Delete all files from a folder (Google Drive) or a bucket
#' (Google Cloud Storage). Caution: This will permanently delete
#' your backup files generated by using ee_as_stars and ee_as_sf.
#'
#' @param name Character. Name of the folder (Drive) or bucket (GCS)
#' to delete all files into.
#' @param type Character. Name of the file storage web service. 'drive'
#' and 'gcs' are supported.
#' @param quiet logical. Suppress info message
#' @return No return value, called for cleaning Google Drive or Google Cloud Storage container.
#' @family ee_clean functions
#'
#' @export
ee_clean_container <- function(name = "rgee_backup",
                               type = "drive",
                               quiet = FALSE) {
  ee_user <- ee_exist_credentials()

  if (type == "drive") {
    ee_check_packages("ee_download_drive", "googledrive")
    if (is.na(ee_user[["drive_cre"]])) {
      stop(
        "Google Drive credentials were not loaded.",
        ' Run ee_Initialize(email = "myemail", drive = TRUE)',
        " to fix it"
      )
    }
    count <- 1
    try_gd_rm <- try(
      expr = googledrive::drive_rm(name, verbose = !quiet),
      silent = TRUE
    )
    while (class(try_gd_rm) == "try-error" & count < 5) {
      try_gd_rm <- try(
        expr = googledrive::drive_rm(name, verbose = !quiet),
        silent = TRUE
      )
      count <- count + 1
    }
  } else if (type == "gcs") {
    # check if googleCloudStorageR is installed
    ee_check_packages("ee_download_gcs", "googleCloudStorageR")

    if (is.na(ee_user[["gcs_cre"]])) {
      stop(
        "Google Drive credentials were not loaded.",
        ' Run ee_Initialize(email = "myemail", gcs = TRUE)',
        " to fix it"
      )
    }
    if (isFALSE(quiet)) {
      googleCloudStorageR::gcs_global_bucket(name)
      buckets <- googleCloudStorageR::gcs_list_objects(bucket = name)
      gcs_todelete <- buckets[["name"]]
      mapply(googleCloudStorageR::gcs_delete_object, gcs_todelete)
    } else {
      suppressMessages(
        googleCloudStorageR::gcs_global_bucket(name)
      )
      suppressMessages(
        buckets <- googleCloudStorageR::gcs_list_objects()
      )
      gcs_todelete <- buckets[["name"]]
      suppressMessages(
        mapply(googleCloudStorageR::gcs_delete_object, gcs_todelete)
      )
    }
  } else {
    stop("type argument invalid.")
  }
  invisible(TRUE)
}
