% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/report_spec.r
\name{write_report}
\alias{write_report}
\title{Write a report to the file system}
\usage{
write_report(x, file_path = NULL, output_type = NULL, preview = NULL)
}
\arguments{
\item{x}{The report object to write.}

\item{file_path}{The file name and path to write the report to.  If supplied,
this parameter overrides the \code{file_path} parameter on the 
\code{create_report} function. Default is NULL.}

\item{output_type}{The output file type.  This parameter will override
the \code{output_type} on the \code{create_report} function.  This 
parameter can be used to output the same report object to 
multiple output types. Default value is NULL, meaning it will not override
the \code{create_report} value.  Valid values are 'TXT', 'RTF', and 'PDF'.}

\item{preview}{Whether to write the entire report, or a report preview.
A report preview is a subset of pages of the report.  The default value is 
NULL, meaning the entire report will be written.  You may also pass 
a number of pages to write.  For example, passing the number 1 will print
the first page, while passing a 5 will print the first five pages.}
}
\value{
The report spec, with settings modified during rendering.  These 
modified settings can sometimes be useful for documentation, and for 
debugging issues with the procedure.
}
\description{
This function writes a report_spec object to the file system, using the
specifications provided in the object.
}
\details{
The function renders the report in the requested format, and writes it
to the location specified in the report \code{file_path} parameter.  Attempts
to write an object that is not of class "report_spec" will generate an error.

The \code{write_report} function is a driver for very complex set of 
rendering functions. The rendering functions
perform most of the advanced functionality of the \strong{reporter} package:
generating spanning headers, page wrapping and breaking, creating stub 
columns, etc.  When things go wrong, they will usually go wrong during this
function call.  For that reason, although this function can be part of 
the pipeline that creates the report object, it is best to call 
\code{write_report} independently, to help isolate any issues from the 
report definition procedure.
}
\examples{
library(reporter)
library(fmtr)
library(magrittr)

# Create temp file path
tmp <- file.path(tempdir(), "beaver2.txt")

# Take Sample of Data
dat <- beaver2[sample(1:100, 15), ]

# Create format for active variable
fmt <- value(condition(x == 0, "No"),
             condition(x == 1, "Yes"))

# Create the table
tbl <- create_table(dat) \%>\% 
  titles("Table 1.0", "BEAVERS Sample Report") \%>\% 
  column_defaults(width = .75) \%>\% 
  define(day, label = "Day", format = "Day \%s") \%>\% 
  define(time, label = "Time") \%>\% 
  define(temp, label = "Temperature", width = 1, format = "\%.1f") \%>\% 
  define(activ,label = "Active", format = fmt) \%>\% 
  footnotes("* NOTE: Data on beaver habits")

# Create the report object
rpt <- create_report(tmp) \%>\% 
  add_content(tbl, align = "left") 

# Write the report to the file system
res <- write_report(rpt)

# Write the modified report object to the console
print(res)

# Write the report to console
writeLines(readLines(tmp, encoding = "UTF-8"))

#                 Table 1.0
#           BEAVERS Sample Report
# 
#      Day      Time  Temperature    Active
# -----------------------------------------
#  Day 307      1020         37.2        No
#  Day 307      1030         37.2        No
#  Day 307       940         36.7        No
#  Day 307      1340         37.1        No
#  Day 307      1410         37.2        No
#  Day 307      1400         37.1        No
#  Day 307      1130         36.9        No
#  Day 307      1140         37.0        No
#  Day 307      1120         37.0        No
#  Day 307      1000         37.1        No
#  Day 307      1250         37.0        No
#  Day 307      2100         37.9       Yes
#  Day 307      1210         37.0        No
#  Day 307      1740         38.0       Yes
#  Day 308       130         37.8       Yes
# 
# * NOTE: Data on beaver habits
}
\seealso{
Other report: 
\code{\link{add_content}()},
\code{\link{create_report}()},
\code{\link{footnotes}()},
\code{\link{options_fixed}()},
\code{\link{page_by}()},
\code{\link{page_footer}()},
\code{\link{page_header}()},
\code{\link{print.report_spec}()},
\code{\link{set_margins}()},
\code{\link{title_header}()},
\code{\link{titles}()}
}
\concept{report}
