#'
#'
#' Multiple starts for Regularized Structural Equation Modeling
#'
#'
#' @param model Lavaan output object. This is a model that was previously
#'        run with any of the lavaan main functions: cfa(), lavaan(), sem(),
#'        or growth(). It also can be from the efaUnrotate() function from
#'        the semTools package. Currently, the parts of the model which cannot
#'        be handled in regsem is the use of multiple group models, missing
#'        other than listwise, thresholds from categorical variable models,
#'        the use of additional estimators other than
#'        ML, most notably WLSMV for categorical variables. Note: the model
#'        does not have to actually run (use do.fit=FALSE), converge etc...
#'        regsem() uses the lavaan object as more of a parser and to get
#'        sample covariance matrix.
#' @param max.try number of starts to try before convergence.
#' @param lambda Penalty value. Note: higher values will result in additional
#'        convergence issues.
#' @param LB lower bound vector. Note: This is very important to specify
#'        when using regularization. It greatly increases the chances of
#'        converging.
#' @param UB Upper bound vector
#' @param block Whether to use block coordinate descent
#' @param full Whether to do full gradient descent or block
#' @param type Penalty type. Options include "none", "lasso", "ridge",
#'        "enet" for the elastic net,
#'        "alasso" for the adaptive lasso, "scad, "mcp",
#'        and "diff_lasso". diff_lasso penalizes the discrepency between
#'        parameter estimates and some pre-specified values. The values
#'        to take the deviation from are specified in diff_par.
#' @param optMethod Solver to use. Recommended options include "nlminb" and
#'        "optimx". Note: for "optimx", the default method is to use nlminb.
#'        This can be changed in subOpt.
#' @param gradFun Gradient function to use. Recommended to use "ram",
#'        which refers to the method specified in von Oertzen & Brick (2014).
#'        The "norm" procedure uses the forward difference method for
#'        calculating the hessian. This is slower and less accurate.
#' @param pars_pen Parameter indicators to penalize. If left NULL, by default,
#'        all parameters in the \emph{A} matrix outside of the intercepts are
#'        penalized when lambda > 0 and type != "none".
#' @param diff_par Parameter values to deviate from. Only used when
#'        type="diff_lasso".
#' @param hessFun Hessian function to use. Recommended to use "ram",
#'        which refers to the method specified in von Oertzen & Brick (2014).
#'        The "norm" procedure uses the forward difference method for
#'        calculating the hessian. This is slower and less accurate.
#' @param tol Tolerance for coordinate descent
#' @param solver Whether to use solver for coord_desc
#' @param solver.maxit Max iterations for solver in coord_desc
#' @param alpha.inc Whether alpha should increase for coord_desc
#' @param step Step size
#' @param momentum Logical for coord_desc
#' @param step.ratio Ratio of step size between A and S. Logical
#' @param verbose Whether to print iteration number.
#' @param warm.start Whether start values are based on previous iteration.
#'        This is not recommended.
#' @param Start2 Provided starting values. Not required
#' @param nlminb.control list of control values to pass to nlminb
#' @keywords multiple optim
#' @export
#' @examples
#' \dontrun{
#' # Note that this is not currently recommend. Use regsem() instead
#' library(regsem)
#' HS <- data.frame(scale(HolzingerSwineford1939[,7:15]))
#' mod <- '
#' f =~ x1 + x2 + x3 + x4 + x5 + x6 + x7 + x8 + x9
#' '
#' outt = cfa(mod,HS,meanstructure=TRUE)
#'
#' fit1 <- multi_optim(outt,max.try=40,
#'                    lambda=0.1,type="lasso",
#'                    gradFun="ram")
#'
#'
#'# growth model
#'model <- ' i =~ 1*t1 + 1*t2 + 1*t3 + 1*t4
#'           s =~ 0*t1 + s1*t2 + s2*t3 + 3*t4 '
#'fit <- growth(model, data=Demo.growth)
#'summary(fit)
#'fitmeasures(fit)

#'fit3 <- multi_optim(fit,lambda=0.2,type="ridge",gradFun="none")
#'summary(fit3)
#'}


multi_optim <- function(model,max.try=10,lambda=0,
                         LB=-Inf,
                        UB=Inf,
                        block=TRUE,
                        full=TRUE,
                        type="none",
                        optMethod="default",
                        gradFun="ram",
                        pars_pen=NULL,
                        diff_par=NULL,
                        hessFun="none",
                        tol=1e-5,
                        solver=FALSE,
                        solver.maxit=50000,
                        alpha.inc=TRUE,
                        step=.5,
                        momentum=FALSE,
                        step.ratio=FALSE,
                        verbose=FALSE,warm.start=FALSE,Start2=NULL,
                        nlminb.control=NULL){


#  if(optMethod=="default" & type=="lasso"){
#    optMethod<-"coord_desc"
#  }

#  if(optMethod=="default" & type!="lasso"){
#    optMethod <- "nlminb"
#  }

 # warning("Note it is not currently recommended to use multi_optim")

#  if(gradFun=="norm"){
#    stop("Only recommended grad function is ram or none at this time")
#  }

#  if(type=="ridge" & gradFun != "none"){
#    warning("At this time, only gradFun=none recommended with ridge penalties")
#  }

#  if(type=="lasso" & gradFun != "ram"){
#    warning("At this time, only gradFun=ram recommended with lasso penalties")
#  }


#if(warm.start==TRUE){
#  stop("warm start not currently functioning")
#}
    if(warm.start==TRUE & is.null(Start2)){
     # fit99 <- suppressWarnings(regsem(model,lambda=lambda,type=type,optMethod=optMethod,
      #                                 Start=start.optim,gradFun=gradFun,hessFun=hessFun,max.iter=max.iter,
      #                                 LB=LB,UB=UB,pars_pen=pars_pen,diff_par=diff_par,tol=tol))
      Start2 = rep(0.5,length(extractMatrices(model)$parameters)) +
        rnorm(length(extractMatrices(model)$parameters),0,0.05)
#
 #     if(length(start.vals) != length(extractMatrices(model)$parameters)){
 #       start.vals = rep(0.5,length(extractMatrices(model)$parameters)) +
 #         rnorm(length(extractMatrices(model)$parameters),0,0.05)
  #    }

    }

  mats <- extractMatrices(model)

  val1 = max(mats$A)
  val2 = max(mats$S) - max(mats$A)

  sss = seq(1,max.try)

    mult_run <- function(model,n.try=1,lambda,LB=-Inf,tol,
                         UB=Inf,
                         block,
                         full,
                         type,optMethod,warm.start,
                         solver,
                         solver.maxit,
                         alpha.inc,
                         step,
                         momentum,
                         step.ratio,
                         gradFun,n.optim,pars_pen,nlminb.control,
                         diff_par,hessFun,Start2){
      mtt = matrix(NA,n.try,3)
      mtt[,3] = round(runif(n.try,1,99999))
      start.optim=NULL
      n.optim = 0
      while(n.optim < n.try){
        n.optim=n.optim+1
        set.seed(mtt[n.optim,3])

        if(warm.start==FALSE){
          if(is.null(Start2)){



            #start.optim = c(rep(0,val1) + rnorm(val1,0,0.1),abs(rep(0.5,val2) + rnorm(val2,0,0.1)))
            start.optim=mats$parameters

          }else{
            start.optim = Start2
          }
        }else if(warm.start==TRUE){
          start.optim= mats$parameters
        }

        #else if(warm.start==TRUE){
         # start.optim <- rep(0,length(start.vals))
         # for(i in 1:length(start.vals)){
         #   start.optim[i] <- as.numeric(start.vals[i]) + rnorm(1,0,0.02)
         # }
       # }


        fit1 <- suppressWarnings(try(regsem(model,lambda=lambda,type=type,optMethod=optMethod,
                                            Start=start.optim,gradFun=gradFun,hessFun=hessFun,
                                            nlminb.control=nlminb.control,tol=tol,
                                            solver=solver,
                                            block=block,
                                            full=full,
                                            solver.maxit=solver.maxit,
                                            alpha.inc=alpha.inc,
                                            step=step,
                                            momentum=momentum,
                                            step.ratio=step.ratio,
                                            LB=LB,UB=UB,pars_pen=pars_pen,diff_par=diff_par),silent=T))



        if(inherits(fit1, "try-error")) {
          mtt[n.optim,1] = NA
          mtt[n.optim,2] = NA
          start.vals = NULL

          if(warm.start == TRUE){
            start.vals <- mats$parameters +
              rnorm(length(mats$parameters),0,0.05)
          }
        }else{
          start.vals = NULL

          if(warm.start==TRUE){
            start.vals <- as.numeric(fit1$coefficients) + rnorm(length(mats$parameters),0,0.0001)
          }

          if(optMethod=="nlminb"){
            mtt[n.optim,1] = fit1$out$objective
            mtt[n.optim,2] = fit1$out$convergence
          }else{
            #print(fit1$out$value)
            mtt[n.optim,1] = fit1$out$value
            mtt[n.optim,2] = fit1$out$convergence
          }
        }
    }
    ret =  list(mtt=mtt,start.vals=start.vals,fit1=fit1)
    ret
  }


iter.optim = 0
while(iter.optim < max.try){
iter.optim = iter.optim + 1



    ret.mult = mult_run(model,n.try=1,lambda=lambda,LB,UB,type,warm.start=warm.start,
                        nlminb.control=nlminb.control,tol=tol,
                        solver=solver,
                        block=block,
                        full=full,
                        solver.maxit=solver.maxit,
                        alpha.inc=alpha.inc,
                        step=step,
                        momentum=momentum,
                        step.ratio=step.ratio,
                    optMethod=optMethod,gradFun=gradFun,n.optim=iter.optim,Start2=Start2,
                    pars_pen=pars_pen,diff_par=diff_par,hessFun=hessFun)



    if(warm.start == TRUE & ret.mult$mtt[1,1] > 0){
      Start2 = ret.mult$start.vals +
        c(rep(0,val1) + rnorm(val1,0,0.01),abs(rep(0,val2) + rnorm(val2,0,0.001)))
    }else{
      Start2 = c(rep(0,val1) + rnorm(val1,0,0.1),abs(rep(0.5,val2) + rnorm(val2,0,0.1)))
    }

    outt = ret.mult$mtt


    if(verbose==TRUE) print(c(iter.optim,outt[,1],outt[,2]))

      if(all(is.na(outt[,2])==TRUE)){
        return
      }else if(any(na.omit(outt[,2]) == 0)){
        if(any(is.na(outt[,1]) == FALSE & outt[,1] < 999999 & outt[,1] > 0)){
       # row = which(outt[,1] == min(outt[which(is.na(outt[,1]) == FALSE & outt[,1] > 0 & outt[,2] == 0)]))[1]
       # set.seed(outt[row,3])
      #  if(warm.start==FALSE){
       #   start.optim = rep(0.5,length(extractMatrices(model)$parameters)) +
       #     rnorm(length(extractMatrices(model)$parameters),0,0.05)
       # }else if(warm.start==TRUE){
       #   start.optim = Start2
       # }

        #fit1 <- suppressWarnings(regsem(model,lambda=lambda,type=type,optMethod=optMethod,
        #               Start=start.optim,gradFun=gradFun,hessFun=hessFun,max.iter=max.iter,
         #              LB=LB,UB=UB,pars_pen=pars_pen,diff_par=diff_par,tol=tol))
        return(ret.mult$fit1)
        break
        }else{
          return
        }
      }else{
        return
      }
    }
   # if(exists("fit1")==FALSE){

      if(warm.start==TRUE){
        Start=Start2
      }else{
        Start="default"
      }
      fit1 <- suppressWarnings(regsem(model,lambda=lambda,type=type,optMethod=optMethod,
                     Start=Start,gradFun=gradFun,hessFun=hessFun,
                     nlminb.control=nlminb.control,tol=tol,
                     solver=solver,
                     block=block,
                     full=full,
                     solver.maxit=solver.maxit,
                     alpha.inc=alpha.inc,
                     step=step,
                     momentum=momentum,
                     step.ratio=step.ratio,
                     LB=LB,UB=UB,pars_pen=pars_pen,diff_par=diff_par))

        fit1$convergence <- 99
        return(fit1)
   # }



    if(fit1$convergence != 0){
      warning("WARNING: Model did not converge! It is recommended to increase max.try")
    }

}



