% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fractional.operators.R
\name{matern.operators}
\alias{matern.operators}
\title{Rational approximations of stationary Gaussian Matern random fields}
\usage{
matern.operators(
  kappa,
  sigma,
  nu,
  G = NULL,
  C = NULL,
  d = NULL,
  mesh = NULL,
  m = 1,
  type = c("covariance", "operator")
)
}
\arguments{
\item{kappa}{Range parameter of the covariance function.}

\item{sigma}{Standard deviation of the covariance function.}

\item{nu}{Shape parameter of the covariance function.}

\item{G}{The stiffness matrix of a finite element discretization of the domain of interest. Does not need to be given if \code{mesh} is used.}

\item{C}{The mass matrix of a finite element discretization of the domain of interest. Does not need to be given if \code{mesh} is used.}

\item{d}{The dimension of the domain. Does not need to be given if \code{mesh} is used.}

\item{mesh}{An optional inla mesh. \code{d}, \code{C} and \code{G} must be given if \code{mesh} is not given.}

\item{m}{The order of the rational approximation, which needs to be a positive integer.
The default value is 1.}

\item{type}{The type of the rational approximation. The options are "covariance"
and "operator". The default is "covariance".}
}
\value{
If \code{type} is "covariance", then \code{matern.operators} 
returns an object of class "CBrSPDEobj". 
This object is a list containing the
following quantities:
\item{C}{The mass lumped mass matrix.}
\item{Ci}{The inverse of \code{C}.}
\item{GCi}{The stiffness matrix G times \code{Ci}}
\item{Gk}{The stiffness matrix G along with the higher-order FEM-related matrices G2, G3, etc.}
\item{fem_mesh_matrices}{A list containing the mass lumped mass matrix, the stiffness matrix and 
the higher-order FEM-related matrices.}
\item{m}{The order of the rational approximation.}
\item{alpha}{The fractional power of the precision operator.}
\item{type}{String indicating the type of approximation.}
\item{d}{The dimension of the domain.}
\item{nu}{Shape parameter of the covariance function.}
\item{kappa}{Range parameter of the covariance function}
\item{tau}{Scale parameter of the covariance function.}
\item{sigma}{Standard deviation of the covariance function.}
\item{type}{String indicating the type of approximation.}

If \code{type} is "operator", then \code{matern.operators}
 returns an object of class "rSPDEobj". This object contains the
quantities listed in the output of \code{\link{fractional.operators}},
the \code{G} matrix, the dimension of the domain, as well as the 
parameters of the covariance function.
}
\description{
\code{matern.operators} is used for computing a rational SPDE approximation of a stationary Gaussian random
fields on \eqn{R^d} with a Matern covariance function
\deqn{C(h) = \frac{\sigma^2}{2^{\nu-1}\Gamma(\nu)}(\kappa h)^\nu K_\nu(\kappa h)}{C(h) =
(\sigma^2/(2^{\nu-1}\Gamma(\nu))(\kappa h)^\nu K_\nu(\kappa h).}
}
\details{
If \code{type} is "covariance", we use the 
covariance-based rational approximation of the fractional operator.
In the SPDE approach, we model \eqn{u} as the solution of the following SPDE: 
\deqn{L^{\alpha/2}(\tau u) = \mathcal{W},}
where 
\eqn{L  = -\Delta +\kappa^2 I} and \eqn{\mathcal{W}} is the standard Gaussian white noise. 
The covariance operator of \eqn{u} is given by \eqn{L^{-\alpha}}. 
Now, let \eqn{L_h} be a finite-element approximation of \eqn{L}. We can use 
a rational approximation of order \eqn{m} on \eqn{L_h^{-\alpha}} to
obtain the following approximation:
\deqn{L_{h,m}^{-\alpha} = L_h^{-m_\alpha} p(L_h^{-1})q(L_h^{-1})^{-1},}
where \eqn{m_\alpha = \lfloor \alpha\rfloor}, \eqn{p} and \eqn{q} are polynomials arising from such rational approximation.
From this approximation we construct an approximate precision matrix for \eqn{u}.

If \code{type} is "operator", the approximation is based on a 
rational approximation of the fractional operator
\eqn{(\kappa^2 -\Delta)^\beta}, where \eqn{\beta = (\nu + d/2)/2}.
This results in an approximate model of the form \deqn{P_l u(s) = P_r W,}
where \eqn{P_j = p_j(L)} are non-fractional operators defined in terms of polynomials \eqn{p_j} for
\eqn{j=l,r}. The order of \eqn{p_r} is given by \code{m} and the order of \eqn{p_l} is \eqn{m + m_\beta}
where \eqn{m_\beta} is the integer part of \eqn{\beta} if \eqn{\beta>1} and
\eqn{m_\beta = 1} otherwise.

The discrete approximation can be written as \eqn{u = P_r x} where \eqn{x \sim N(0,Q^{-1})}{x ~ N(0,Q^{-1})}
and \eqn{Q = P_l^T C^{-1} P_l}. Note that the matrices \eqn{P_r} and \eqn{Q} may be be ill-conditioned for \eqn{m>1}.
In this case, the methods in \code{\link{operator.operations}} should be used for operations
involving the matrices, since these methods are more numerically stable.
}
\examples{
#Compute the covariance-based rational approximation of a 
#Gaussian process with a Matern covariance function on R
kappa <- 10
sigma <- 1
nu <- 0.8

#create mass and stiffness matrices for a FEM discretization
nobs = 101
x <- seq(from = 0, to = 1, length.out = 101)
fem <- rSPDE.fem1d(x)

#compute rational approximation of covariance function at 0.5
op_cov <- matern.operators(C=fem$C, G=fem$G,nu=nu,
kappa=kappa,sigma=sigma,d=1,m=2)

v = t(rSPDE.A1d(x,0.5))
#Compute the precision matrix
Q <- rspde.matern.precision(kappa=kappa,nu=nu,sigma=sigma,
rspde_order=2,d=1,fem_mesh_matrices = op_cov$fem_mesh_matrices)
#A matrix here is the identity matrix
A <- Diagonal(nobs)
#We need to concatenate 3 A's since we are doing a covariance-based rational
#approximation of order 2
Abar <- cbind(A,A,A)
w <- rbind(v,v,v)
#The approximate covariance function:
c_cov.approx <- (Abar)\%*\%solve(Q,w)
c.true <- folded.matern.covariance(rep(0.5,length(x)),abs(x), kappa, nu, sigma)

#plot the result and compare with the true Matern covariance
plot(x, c.true, type = "l", ylab = "C(h)",
     xlab="h", main = "Matern covariance and rational approximations")
lines(x, c_cov.approx, col = 2)


#Compute the operator-based rational approximation of a Gaussian 
#process with a Matern covariance function on R
kappa <- 10
sigma <- 1
nu <- 0.8

#create mass and stiffness matrices for a FEM discretization
x <- seq(from = 0, to = 1, length.out = 101)
fem <- rSPDE.fem1d(x)

#compute rational approximation of covariance function at 0.5
op <- matern.operators(kappa = kappa, sigma = sigma, nu = nu,
                       G = fem$G, C = fem$C, d = 1,
                       type="operator")
                       
v = t(rSPDE.A1d(x,0.5))
c.approx = Sigma.mult(op,v)
c.true <- folded.matern.covariance(rep(0.5,length(x)),abs(x), kappa, nu, sigma)

#plot the result and compare with the true Matern covariance
plot(x, c.true, type = "l", ylab = "C(h)",
     xlab="h", main = "Matern covariance and rational approximation")
lines(x,c.approx,col=2)
}
\seealso{
\code{\link{fractional.operators}}, \code{\link{spde.matern.operators}},
\code{\link{matern.operators}}
}
