library(provParseR)
library(testthat)

# Check that null is returned when there is no provenance
expect_equal (is.null (get.environment(NULL)), TRUE)

## Loading test data
test.data <- system.file("testdata", "ddg.json", package = "provParseR")
prov <- prov.parse(test.data)

context("Environment access function")
envi.df <- get.environment(prov)
expect_match(class(envi.df), "data.frame")
expect_match(typeof(envi.df$value), "character")
expect_equal(nrow(envi.df), 12)
expect_equal(ncol(envi.df), 2)
expect_equal (envi.df$label, c("name", "architecture", "operatingSystem", "language", "rVersion",
        "script", "scriptTimeStamp", "workingDirectory", "ddgDirectory", "ddgTimeStamp", 
        "rdatatrackerVersion", "hashAlgorithm"))

context("Procedure nodes access function")
proc.df <- get.proc.nodes(prov)
expect_match(class(proc.df), "data.frame")
expect_match(typeof(proc.df$name), "character")
expect_match(typeof(proc.df$type), "character")
expect_match(typeof(proc.df$elapsedTime), "double")
expect_match(typeof(proc.df$scriptNum), "integer")
expect_match(typeof(proc.df$startLine), "integer")
expect_match(typeof(proc.df$startCol), "integer")
expect_match(typeof(proc.df$endLine), "integer")
expect_match(typeof(proc.df$endCol), "integer")
expect_equal(nrow(proc.df), 37)
expect_equal(ncol(proc.df), 9)

context("Data nodes access function")
data.df <- get.data.nodes(prov)
expect_match(class(data.df), "data.frame")
expect_match(typeof(data.df$name), "character")
expect_match(typeof(data.df$value), "character")
expect_match(typeof(data.df$valType), "character")
expect_match(typeof(data.df$type), "character")
expect_match(typeof(data.df$scope), "character")
expect_match(typeof(data.df$fromEnv), "logical")
expect_match(typeof(data.df$hash), "character")
expect_match(typeof(data.df$timestamp), "character")
expect_match(typeof(data.df$location), "character")
expect_equal(nrow(data.df), 16)
expect_equal(ncol(data.df), 10)

context("Exception nodes access function")
errors.df <- get.error.nodes(prov)
expect_equal(nrow(errors.df), 0)

context("Function nodes access function")
func.df <- get.func.nodes(prov)
expect_match(class(func.df), "data.frame")
expect_match(typeof(func.df$name), "character")
expect_equal(nrow(func.df), 8)
expect_equal(ncol(func.df), 2)
expect_equal (func.df$name, c("read.csv", "head", "tail", "str", "plot", "line", "abline", "coef"))

context("Procedure-to-procedure edges access function")
proc.proc.df <- get.proc.proc(prov)
expect_match(class(proc.proc.df), "data.frame")
expect_match(typeof(proc.proc.df$informant), "character")
expect_match(typeof(proc.proc.df$informed), "character")
expect_equal(nrow(proc.proc.df), 36)
expect_equal(ncol(proc.proc.df), 3)

context("Data-to-procedure edges access function")
data.proc.df <- get.data.proc(prov)
expect_match(class(data.proc.df), "data.frame")
expect_match(typeof(data.proc.df$entity), "character")
expect_match(typeof(data.proc.df$activity), "character")
expect_equal(nrow(data.proc.df), 39)
expect_equal(ncol(data.proc.df), 3)

context("Procedure-to-data edges access function")
proc.data.df <- get.proc.data(prov)
expect_match(class(proc.data.df), "data.frame")
expect_match(typeof(proc.data.df$entity), "character")
expect_match(typeof(proc.data.df$activity), "character")
expect_equal(nrow(proc.data.df), 15)
expect_equal(ncol(proc.data.df), 3)

context("Function-to-procedure edges access function")
func.proc.df <- get.func.proc(prov)
expect_match(class(func.proc.df), "data.frame")
expect_match(typeof(func.proc.df$"function"), "character")
expect_match(typeof(func.proc.df$activity), "character")
expect_equal(nrow(func.proc.df), 8)
expect_equal(ncol(func.proc.df), 3)

context("Function-library group nodes access function")
func.lib.df <- get.func.lib(prov)
expect_match(class(func.lib.df), "data.frame")
expect_match(typeof(func.lib.df$library), "character")
expect_match(typeof(func.lib.df$"function"), "character")
expect_equal(nrow(func.lib.df), 8)
expect_equal(ncol(func.lib.df), 3)

context("Library nodes access function")
libs.df <- get.libs(prov)
expect_match(class(libs.df), "data.frame")
expect_match(typeof(libs.df$name), "character")
expect_match(typeof(libs.df$version), "character")
expect_equal(nrow(libs.df), 13)
expect_equal(ncol(libs.df), 3)

context("Scripts access function")
scripts.df <- get.scripts(prov)
expect_match(class(scripts.df), "data.frame")
expect_equal (nrow (scripts.df), 1)
expect_match(typeof(scripts.df$script), "character")
expect_match(typeof(scripts.df$timestamp), "character")
expect_equal(ncol(scripts.df), 2)

context ("Input files")
input.files <- get.input.files (prov, only.files=TRUE)
expect_equal (nrow (input.files), 0)

context ("URLs")
urls <- get.urls (prov)
expect_equal (nrow (urls), 1)
expect_equal (urls$name, "http://harvardforest.fas.harvard.edu/sites/harvardforest.fas.harvard.edu/files/weather/metsta.dat")

context ("Output files")
output.files <- get.output.files (prov)
expect_equal (nrow (output.files), 1)
expect_setequal (output.files$name, "dev.off.16.pdf")

context ("Variables set")
variables.set <- get.variables.set (prov)
expect_equal (nrow (variables.set), 14)

context ("Variables used")
variables.used <- get.variables.used (prov)
expect_equal (nrow (variables.used), 14)


## If input is a string

context("String prov")
prov <- prov.parse(readLines(test.data), isFile = FALSE)

context("String prov - Environment access function")
envi.df <- get.environment(prov)
expect_match(class(envi.df), "data.frame")
expect_match(typeof(envi.df$value), "character")
expect_equal(nrow(envi.df), 12)
expect_equal(ncol(envi.df), 2)

context("String prov - Procedure nodes access function")
proc.df <- get.proc.nodes(prov)
expect_match(class(proc.df), "data.frame")
expect_match(typeof(proc.df$name), "character")
expect_match(typeof(proc.df$type), "character")
expect_match(typeof(proc.df$elapsedTime), "double")
expect_match(typeof(proc.df$scriptNum), "integer")
expect_match(typeof(proc.df$startLine), "integer")
expect_match(typeof(proc.df$startCol), "integer")
expect_match(typeof(proc.df$endLine), "integer")
expect_match(typeof(proc.df$endCol), "integer")
expect_equal(nrow(proc.df), 37)
expect_equal(ncol(proc.df), 9)

