#' Permanent Random Number Sampling in R
#'
#' @description
#' This package provides two functions for drawing stratified
#' PRN-assisted samples: \code{srs} and \code{pps}. The former --
#' simple random sampling -- assumes that each unit \eqn{k} in a given
#' stratum \eqn{h} is equally likely to be sampled, with inclusion
#' probability \deqn{\pi_k = \frac{n_h}{N_h}} for each stratum \eqn{h}.
#' The function then samples the \eqn{n_h} elements with the smallest PRN's,
#' for each stratum \eqn{h}.
#'
#' The latter -- Pareto \eqn{\pi ps} sampling -- assumes that large units are
#' more likely to be sampled than small units. The function approximates this
#' unknown inclusion probability as
#' \deqn{\lambda_k = n_h \frac{x_k}{\sum_{i=1}^{n_h} x_i},} where
#' \eqn{x_k} is a size measure, and samples the \eqn{n_h} elements with the
#' smallest values of
#' \deqn{Q_k = \frac{PRN_k(1 - \lambda_k)}{\lambda_k(1 - PRN_k)},}
#' for each stratum \eqn{h}.
#'
#' These two functions can be run standalone or via the wrapper function
#' \code{samp}. Input to the functions is the sampling frame, stratification
#' information and PRN's given as variables on the frame, and in the case for
#' \code{pps} also a size measure given as variable on the frame. Output is a
#' copy of the sampling frame containing sampling information, and in the case
#' for \code{pps} also containing \eqn{\lambda} and \eqn{Q}.
#'
#' Provided is also a function \code{transformprn} via which it is possible to
#' select where to start counting and in which direction when enumerating the
#' PRN's in the sampling routines. This is done by specifying starting point
#' and direction to \code{transformprn} and then calling \code{srs} or
#' \code{pps} on its output.
#'
#' Finally, an example dataset is provided that can be used to illustrate the
#' functionality of the package.
#'
#' @references Lindblom, A. (2014). "On Precision in Estimates of Change over
#' Time where Samples are Positively Coordinated by Permanent Random Numbers."
#' \emph{Journal of Official Statistics}, vol.30, no.4, 2014, pp.773-785.
#' https://doi.org/10.2478/jos-2014-0047.
#'
#' @seealso \link{srs}, \link{pps}, \link{samp}, \link{transformprn},
#' \link{ExampleData}
#'
#' @name prnsamplr-package
#'
#' @examples
#' dfSRS <- srs(
#'   frame = ExampleData,
#'   nsamp = ~nsample,
#'   stratid = ~stratum,
#'   prn = ~rands
#' )
#'
#' dfPPS <- pps(
#'   frame = ExampleData,
#'   nsamp = ~nsample,
#'   stratid = ~stratum,
#'   prn = ~rands,
#'   size = ~sizeM
#' )
#'
#' dfPRN <- transformprn(
#'   frame = ExampleData,
#'   prn = ~rands,
#'   direction = "U",
#'   start = 0.2
#' )
## usethis namespace: start
#' @importFrom stats aggregate setNames
## usethis namespace: end
"_PACKAGE"
