% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_min_shortfall_objective.R
\name{add_min_shortfall_objective}
\alias{add_min_shortfall_objective}
\title{Add minimum shortfall objective}
\usage{
add_min_shortfall_objective(x, budget)
}
\arguments{
\item{x}{\code{\link{ConservationProblem-class}} object.}

\item{budget}{\code{numeric} value specifying the maximum expenditure of
the prioritization. For problems with multiple zones, the argument
to \code{budget} can be a single \code{numeric} value to specify a budget
for the entire solution or a \code{numeric} \code{vector} to specify
a budget for each each management zone.}
}
\value{
\code{\link{ConservationProblem-class}} object with the objective
  added to it.
}
\description{
Set the objective of a conservation planning \code{\link{problem}} to
minimize the shortfall for as many targets as possible while ensuring that
the cost of the solution does not exceed a budget.
}
\details{
A problem objective is used to specify the overall goal of the
  conservation planning problem. Please note that all conservation
  planning problems formulated in the \pkg{prioritizr} package require the
  addition of objectives---failing to do so will return an error
  message when attempting to solve problem.

  The minimum shortfall representation objective aims to
  find the set of planning units that minimize the shortfall for the
  representation targets---that is, the fraction of each target that
  remains unmet---for as many features as possible while staying within a
  fixed budget (inspired by Table 1, equation IV, Arponen \emph{et al.}
  2005). Additionally, weights can be used
   to favor the representation of certain features over other features (see
  \code{\link{add_feature_weights}}.

  The minimum shortfall objective for the reserve design problem can be
  expressed mathematically for a set of planning units (\eqn{I}{I} indexed by
  \eqn{i}{i}) and a set of features (\eqn{J}{J} indexed by \eqn{j}{j}) as:

  \deqn{\mathit{Minimize} \space  \sum_{j = 1}^{J} w_j \frac{y_j}{t_j} \\
  \mathit{subject \space to} \\
  \sum_{i = 1}^{I} x_i r_{ij} + y_j \geq t_j \forall j \in J \\
  \sum_{i = 1}^{I} x_i c_i \leq B}{
  Minimize sum_j^J wj * (yj / tj) subject to
  sum_i^I (xi * rij) + yj >= tj for all j in J &
  sum_i^I (xi * ci) <= B}

  Here, \eqn{x_i}{xi} is the \code{\link{decisions}} variable (e.g.
  specifying whether planning unit \eqn{i}{i} has been selected (1) or not
  (0)), \eqn{r_{ij}}{rij} is the amount of feature \eqn{j}{j} in planning
  unit \eqn{i}{i}, \eqn{t_j}{tj} is the representation target for feature
  \eqn{j}{j}, \eqn{y_j}{yj} denotes the representation shortfall for
  the target \eqn{t_j}{tj} for feature \eqn{j}{j}, and \eqn{w_j}{wj} is the
  weight for feature \eqn{j}{j} (defaults to 1 for all features; see
  \code{\link{add_feature_weights}} to specify weights). Additionally,
  \eqn{B}{B} is the budget allocated for the solution, \eqn{c_i}{ci} is the
  cost of planning unit \eqn{i}{i}. Note that \eqn{y_j}{yj} is a continuous
  variable bounded between zero and infinity, and denotes the shortfall
  for target \eqn{j}{j}.
}
\examples{
# load data
data(sim_pu_raster, sim_pu_zones_stack, sim_features, sim_features_zones)

# create problem with minimum shortfall objective
p1 <- problem(sim_pu_raster, sim_features) \%>\%
      add_min_shortfall_objective(1800) \%>\%
      add_relative_targets(0.1) \%>\%
      add_binary_decisions()
\donttest{
# solve problem
s1 <- solve(p1)

# plot solution
plot(s1, main = "solution", axes = FALSE, box = FALSE)
}

# create multi-zone problem with minimum shortfall objective,
# with 10 \% representation targets for each feature, and set
# a budget such that the total maximum expenditure in all zones
# cannot exceed 3000
p2 <- problem(sim_pu_zones_stack, sim_features_zones) \%>\%
      add_min_shortfall_objective(3000) \%>\%
      add_relative_targets(matrix(0.1, ncol = 3, nrow = 5)) \%>\%
      add_binary_decisions()
\donttest{
# solve problem
s2 <- solve(p2)

# plot solution
plot(category_layer(s2), main = "solution", axes = FALSE, box = FALSE)
}
# create multi-zone problem with minimum shortfall objective,
# with 10 \% representation targets for each feature, and set
# separate budgets for each management zone
p3 <- problem(sim_pu_zones_stack, sim_features_zones) \%>\%
      add_min_shortfall_objective(c(3000, 3000, 3000)) \%>\%
      add_relative_targets(matrix(0.1, ncol = 3, nrow = 5)) \%>\%
      add_binary_decisions()
\donttest{
# solve problem
s3 <- solve(p3)

# plot solution
plot(category_layer(s3), main = "solution", axes = FALSE, box = FALSE)
}
}
\references{
Arponen A, Heikkinen RK, Thomas CD, and Moilanen A (2005) The value of
biodiversity in reserve selection: representation, species weighting, and
benefit functions. \emph{Conservation Biology}, 19: 2009--2014.
}
\seealso{
\code{\link{add_feature_weights}}, \code{\link{objectives}}.
}
