% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/setup.R
\name{study_parameters}
\alias{study_parameters}
\title{Setup study parameters}
\usage{
study_parameters(n1, n2, n3 = 1, T_end = NULL, fixed_intercept = 0,
  fixed_slope = 0, sigma_subject_intercept = NULL,
  sigma_subject_slope = NULL, sigma_cluster_intercept = NULL,
  sigma_cluster_slope = NULL, sigma_error = NULL, cor_subject = 0,
  cor_cluster = 0, cor_within = 0, var_ratio = NULL, icc_slope = NULL,
  icc_pre_subject = NULL, icc_pre_cluster = NULL, cohend = 0,
  partially_nested = FALSE, dropout = 0, deterministic_dropout = TRUE)
}
\arguments{
\item{n1}{Number of level 1 units, e.g. measurements per subject.}

\item{n2}{Number of level 2 units per level 3 unit, e.g. subjects per cluster.
Unbalanced cluster sizes are supported, see \code{\link{unequal_clusters}}.}

\item{n3}{Number of level 3 units per treatment, can be different in each
treatment arm, see \code{\link{per_treatment}}.}

\item{T_end}{Time point of the last measurement. If \code{NULL} it will be set
to \code{n1 - 1}.}

\item{fixed_intercept}{Average baseline value, assumed to be equal for both groups.}

\item{fixed_slope}{Overall change per unit time, in the control group.}

\item{sigma_subject_intercept}{Subject-level random intercept.}

\item{sigma_subject_slope}{Subject-level random slope.}

\item{sigma_cluster_intercept}{Cluster-level random intercept.}

\item{sigma_cluster_slope}{Cluster-level random slope.}

\item{sigma_error}{Within-subjects (residual) variation.}

\item{cor_subject}{Correlation between the subject-level random intercept
and slopes.}

\item{cor_cluster}{Correlation between the cluster-level random intercept
and slopes.}

\item{cor_within}{Correlation of the level 1 residual. Currently ignored in
the analytical power calculations.}

\item{var_ratio}{Ratio of the random
slope variance to the within-subject variance.}

\item{icc_slope}{Proportion of slope variance
at the cluster level.}

\item{icc_pre_subject}{Amount of baseline
variance at the subject level. N.B. this is not really an ICC.}

\item{icc_pre_cluster}{Amount of baseline
variance at the cluster level.}

\item{cohend}{Standardized between-groups treatment effect at the last time point,
see \emph{Details}.}

\item{partially_nested}{\code{logical}; indicates if there's clustering in both
arms or only in the treatment arm.}

\item{dropout}{Dropout process, see \code{\link{dropout_weibull}} or
\code{\link{dropout_manual}}. Assumed to be 0 if \code{NULL}.}

\item{deterministic_dropout}{\code{logical}; if \code{FALSE} the input to
\code{dropout} will be treated as stochastic and dropout will sampled
from a multinomial distribution.}
}
\value{
A \code{list} or \code{data.frame} of parameters values, either of
class \code{plcp} or \code{plcp_muli} if multiple parameters are compared.
}
\description{
Setup the parameters for calculating power for longitudinal multilevel studies
comparing two groups. Ordinary two-level models (subjects with repeated measures),
and longitudinal three-level models with clustering due to therapists, schools, provider etc,
are supported. Random slopes at the subject level and cluster level are
possible. Cluster sizes can be unbalanced, and vary by treatment.
Partially-nested designs are supported. Missing data can also be accounted
for.
}
\details{
\bold{Comparing a combination of parameter values}

It is possible to setup a grid of parameter combinations by entering the values
as vectors. All unique combinations of the inputs will be returned. This is
useful if you want see how different values of the parameters affect power.
See also the convenience function \code{\link{get_power_table}}.

\bold{Standardized and unstandardized inputs}

All parameters of the models can be specified. However, many of the raw
parameter values in a multilevel/LMM do no directly affect the power of the
test of the \code{treatment:time}-coefficient. Power will depend greatly on the relative
size of the parameters, therefore, it is possible to setup your calculations
using only standardized inputs, or by a combination of raw inputs and
standardized inputs. For instance, if \code{sigma_subject_slope} and
\code{icc_slope} is specified, the \code{sigma_cluster_slope} will be
solved for. More information regarding the standardized inputs are available in
the two-level and three-level vignettes.

\bold{Cohen's d calculation}

Cohen's \emph{d} is calculated by using the baseline standard deviation as the standardizer.

\bold{Two- or three-level models}

If either \code{sigma_cluster_slope} or \code{icc_slope} and
 \code{sigma_cluster_intercept} or \code{icc_pre_cluster} is zero or
\code{NULL} it will be assumed a two-level design is wanted.

\bold{Unequal cluster sizes and unbalanced allocation}

It is possible to specify different cluster sizes using
\code{\link{unequal_clusters}}. Cluster sizes can vary between treatment arms
by also using \code{\link{per_treatment}}. The number of clusters per treatment can
also be set by using \code{\link{per_treatment}}. See \code{\link{per_treatment}}
and \code{\link{unequal_clusters}} examples of their use.

\bold{Missing data and dropout}

Accounting for missing data in the power calculations is possible. Currently,
\code{dropout} can be specified using either \code{\link{dropout_weibull}} or
 \code{\link{dropout_manual}}. It is possible to have different dropout
 patterns per treatment group using \code{\link{per_treatment}}. See their
 respective help pages for examples of their use.
}
\examples{
# Three level model with both subject- and cluster-level random slope
# Power calculation using standardized inputs
p <- study_parameters(n1 = 11,
                      n2 = 5,
                      n3 = 4,
                      icc_pre_subject = 0.5,
                      icc_pre_cluster = 0,
                      var_ratio = 0.03,
                      icc_slope = 0.05,
                      cohend = -0.8)

get_power(p)

# The same calculation with all parameters specified directly
p <- study_parameters(n1 = 11,
                      n2 = 5,
                      n3 = 4,
                      T_end = 10,
                      fixed_intercept = 37,
                      fixed_slope = -0.65,
                      sigma_subject_intercept = 2.8,
                      sigma_subject_slope = 0.4726944,
                      sigma_cluster_intercept = 0,
                      sigma_cluster_slope = 0.1084435,
                      sigma_error = 2.8,
                      cor_subject = -0.5,
                      cor_cluster = 0,
                      cohend = -0.8)
get_power(p)

# Standardized and unstandardized inputs
p <- study_parameters(n1 = 11,
                      n2 = 5,
                      n3 = 4,
                      sigma_subject_intercept = 2.8,
                      icc_pre_cluster = 0.07,
                      sigma_subject_slope = 0.47,
                      icc_slope = 0.05,
                      sigma_error = 2.8,
                      cohend = -0.8)

get_power(p)

## Two-level model with subject-level random slope
p <- study_parameters(n1 = 11,
                      n2 = 40,
                      icc_pre_subject = 0.5,
                      var_ratio = 0.03,
                      cohend = -0.8)
get_power(p)

# add missing data
p <- update(p, dropout = dropout_weibull(0.2, 1))
get_power(p)

## Comparing a combination of values
p <- study_parameters(n1 = 11,
                      n2 = c(5, 10),
                      n3 = c(2, 4),
                      icc_pre_subject = 0.5,
                      icc_pre_cluster = 0,
                      var_ratio = 0.03,
                      icc_slope = c(0, 0.05),
                      cohend = c(-0.5, -0.8))

get_power(p)
}
\seealso{
\code{\link{get_power}}, \code{\link{simulate.plcp}}
}
