% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dosing_optim.R
\name{poso_dose_conc}
\alias{poso_dose_conc}
\title{Estimate the optimal dose for a selected target concentration}
\usage{
poso_dose_conc(
  dat = NULL,
  prior_model = NULL,
  tdm = FALSE,
  time_c,
  time_dose = NULL,
  target_conc,
  endpoint = "Cc",
  estim_method = "map",
  nocb = FALSE,
  p = NULL,
  greater_than = TRUE,
  starting_dose = 100,
  interdose_interval = NULL,
  add_dose = NULL,
  duration = 0,
  indiv_param = NULL
)
}
\arguments{
\item{dat}{Dataframe. An individual subject dataset following the
structure of NONMEM/rxode2 event records.}

\item{prior_model}{A \code{posologyr} prior population pharmacokinetics
model, a list of six objects.}

\item{tdm}{A boolean. If \code{TRUE}: estimates the optimal dose for a selected
target concentration at a selected point in time following the events from
\code{dat}, and using Maximum A Posteriori estimation. Setting \code{tdm} to \code{TRUE} causes the
following to occur:
\itemize{
\item the arguments \code{estim_method}, \code{p}, \code{greater_than}, \code{interdose_interval},
\code{add_dose}, \code{indiv_param} and \code{starting_time} are ignored.
}}

\item{time_c}{Numeric. Point in time for which the dose is to be
optimized.}

\item{time_dose}{Numeric. Time when the dose is to be given.}

\item{target_conc}{Numeric. Target concentration.}

\item{endpoint}{Character. The endpoint of the prior model to be optimised
for. The default is "Cc", which is the central concentration.}

\item{estim_method}{A character string. An estimation method to be used for
the individual parameters. The default method "map" is the Maximum A
Posteriori estimation, the method "prior" simulates from the prior
population model, and "sir" uses the Sequential Importance Resampling
algorithm to estimate the a posteriori distribution of the individual
parameters. This argument is ignored if \code{indiv_param} is provided or if
\code{tdm} is set to \code{TRUE}.}

\item{nocb}{A boolean. for time-varying covariates: the next observation
carried backward (nocb) interpolation style, similar to NONMEM.  If
\code{FALSE}, the last observation carried forward (locf) style will be used.
Defaults to \code{FALSE}.}

\item{p}{Numeric. The proportion of the distribution of concentrations to
consider for the optimization. Mandatory for \code{estim_method=sir}. This
argument is ignored if \code{tdm} is set to \code{TRUE}.}

\item{greater_than}{A boolean. If \code{TRUE}: targets a dose leading to a
proportion \code{p} of the concentrations to be greater than \code{target_conc}.
Respectively, lower if \code{FALSE}. This argument is ignored if \code{tdm} is
set to \code{TRUE}.}

\item{starting_dose}{Numeric. Starting dose for the optimization
algorithm.}

\item{interdose_interval}{Numeric. Time for the interdose interval
for multiple dose regimen. Must be provided when add_dose is used. This
argument is ignored if \code{tdm} is set to \code{TRUE}.}

\item{add_dose}{Numeric. Additional doses administered at inter-dose interval
after the first dose. Optional. This argument is ignored if \code{tdm} is set
to \code{TRUE}.}

\item{duration}{Numeric. Duration of infusion, for zero-order
administrations.}

\item{indiv_param}{Optional. A set of individual parameters : THETA,
estimates of ETA, and covariates. This argument is ignored if \code{tdm} is
set to \code{TRUE}.}
}
\value{
A list containing the following components:
\describe{
\item{dose}{Numeric. An optimal dose for the selected target
concentration.}
\item{type_of_estimate}{Character string. The type of estimate of the
individual parameters. Either a point estimate, or a distribution.}
\item{conc_estimate}{A vector of numeric estimates of the conc. Either a
single value (for a point estimate of ETA), or a distribution.}
\item{indiv_param}{A \code{data.frame}. The set of individual parameters used
for the determination of the optimal dose : THETA, estimates of ETA, and
covariates}
}
}
\description{
Estimates the optimal dose for a selected target concentration at a
selected point in time given a population pharmacokinetic model, a set
of individual parameters, a selected point in time, and a target
concentration.
}
\examples{
rxode2::setRxThreads(2L) # limit the number of threads

# model
mod_run001 <- list(
ppk_model = rxode2::rxode({
  centr(0) = 0;
  depot(0) = 0;

  TVCl = THETA_Cl;
  TVVc = THETA_Vc;
  TVKa = THETA_Ka;

  Cl = TVCl*exp(ETA_Cl);
  Vc = TVVc*exp(ETA_Vc);
  Ka = TVKa*exp(ETA_Ka);

  K20 = Cl/Vc;
  Cc = centr/Vc;

  d/dt(depot) = -Ka*depot;
  d/dt(centr) = Ka*depot - K20*centr;
  d/dt(AUC) = Cc;
}),
error_model = function(f,sigma) {
  dv <- cbind(f,1)
  g <- diag(dv\%*\%sigma\%*\%t(dv))
  return(sqrt(g))
},
theta = c(THETA_Cl=4.0, THETA_Vc=70.0, THETA_Ka=1.0),
omega = lotri::lotri({ETA_Cl + ETA_Vc + ETA_Ka ~
    c(0.2,
      0, 0.2,
      0, 0, 0.2)}),
sigma = lotri::lotri({prop + add ~ c(0.05,0.0,0.00)}))
# df_patient01: event table for Patient01, following a 30 minutes intravenous
# infusion
df_patient01 <- data.frame(ID=1,
                        TIME=c(0.0,1.0,14.0),
                        DV=c(NA,25.0,5.5),
                        AMT=c(2000,0,0),
                        EVID=c(1,0,0),
                        DUR=c(0.5,NA,NA))
# estimate the optimal dose to reach a concentration of 80 mg/l
# one hour after starting the 30-minutes infusion
poso_dose_conc(dat=df_patient01,prior_model=mod_run001,
time_c=1,duration=0.5,target_conc=80)

}
