% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/survival_lexis.R
\name{survtab}
\alias{survtab}
\title{Estimate Survival Time Functions}
\usage{
survtab(formula, data, adjust = NULL, breaks = NULL, pophaz = NULL,
  weights = NULL, surv.type = "surv.rel", surv.method = "hazard",
  relsurv.method = "e2", subset = NULL, conf.level = 0.95,
  conf.type = "log-log", verbose = FALSE)
}
\arguments{
\item{formula}{a \code{formula}; e.g. \code{fot ~ sex},
where \code{fot} is the time scale over which you wish to estimate a
survival time function; this
assumes that \code{lex.Xst} in your data is the status variable in the
intended format (almost always right). 
To be explicit, use \code{\link[survival]{Surv}}: e.g. 
\code{Surv(fot, lex.Xst) ~ sex}. 
Variables on the right-hand side of the formula
separated by \code{+} are considered stratifying variables, for which 
estimates are computed separately. May contain usage of \code{adjust()} 
--- see Details and Examples.}

\item{data}{a \code{Lexis} object with at least the survival time scale}

\item{adjust}{can be used as an alternative to passing variables to 
argument \code{formula} within a call to \code{adjust()}; e.g.
\code{adjust = "agegr"}. \link[=flexible_argument]{Flexible input}.}

\item{breaks}{a named list of breaks, e.g.
\code{list(FUT = 0:5)}. If data is not split in advance, \code{breaks}
must at the very least contain a vector of breaks to split the survival time 
scale (mentioned in argument \code{formula}). If data has already been split
(using e.g. \code{\link{splitMulti}}) along at least the used survival time
scale, this may be \code{NULL}. It is generally recommended (and sufficient; 
see Seppa, Dyban and Hakulinen (2015)) to use monthly
intervals where applicable.}

\item{pophaz}{a \code{data.frame} containing
expected hazards for the event of interest to occur. See the
\link[=pophaz]{dedicated help page}. Required when
\code{surv.type = "surv.rel"} or \code{"cif.rel"}. \code{pophaz} must
contain one column named \code{"haz"}, and any number of other columns
identifying levels of variables to do a merge with split data within
\code{survtab}. Some columns may be time scales, which will
allow for the expected hazard to vary by e.g. calendar time and age.}

\item{weights}{typically a list of weights or a \code{character} string
specifying an age group standardization scheme; see
the \link[=direct_standardization]{dedicated help page} 
and examples. NOTE: \code{weights = "internal"} is based on the counts
of persons in follow-up at the start of follow-up (typically T = 0)}

\item{surv.type}{one of \code{'surv.obs'},
\code{'surv.cause'}, \code{'surv.rel'}, 
\code{'cif.obs'} or \code{'cif.rel'}; 
defines what kind of survival time function(s) is/are estimated; see Details}

\item{surv.method}{either \code{'lifetable'} or \code{'hazard'}; determines
the method of calculating survival time functions, where the former computes
ratios such as \code{p = d/(n - n.cens)} 
and the latter utilizes subject-times 
(typically person-years) for hazard estimates such as \code{d/pyrs} 
which are used to compute survival time function estimates.
The former method requires argument \code{n.cens} and the latter 
argument \code{pyrs} to be supplied.}

\item{relsurv.method}{either \code{'e2'} or \code{'pp'}; 
defines whether to compute relative survival using the
EdererII method or using Pohar-Perme weighting;
ignored if \code{surv.type != "surv.rel"}}

\item{subset}{a logical condition; e.g. \code{subset = sex == 1}; 
subsets the data before computations}

\item{conf.level}{confidence level used in confidence intervals; 
e.g. \code{0.95} for 95 percent confidence intervals}

\item{conf.type}{character string; must be one of \code{"plain"}, 
\code{"log-log"} and \code{"log"}; 
defines the transformation used on the survival time
function to yield confidence 
intervals via the delta method}

\item{verbose}{logical; if \code{TRUE}, the function is chatty and
returns some messages and timings along the process}
}
\value{
Returns a table of life time function values and other 
information with survival intervals as rows.
Returns some of the following estimates of survival time functions:

\itemize{
 \item \code{surv.obs} - observed (raw, overall) survival
 \item \code{surv.obs.K} - observed cause-specific survival for cause K
 \item \code{CIF_k} - cumulative incidence function for cause \code{k}
 \item \code{CIF.rel} - cumulative incidence function using excess cases
 \item \code{r.e2} -  relative survival, EdererII
 \item \code{r.pp} -  relative survival, Pohar-Perme weighted
}
The suffix \code{.as} implies adjusted estimates, and \code{.lo} and
\code{.hi} imply lower and upper confidence limits, respectively. 
The prefix \code{SE.} stands for standard error.
}
\description{
This function estimates survival time functions: survival, 
relative/net survival, and crude/absolute risk functions (CIF).
}
\section{Basics}{


This function computes interval-based estimates of survival time functions,
where the intervals are set by the user. For product-limit-based
estimation see packages \pkg{survival} and \pkg{relsurv}.

if \code{surv.type = 'surv.obs'}, only 'raw' observed survival 
is estimated over the chosen time intervals. With
\code{surv.type = 'surv.rel'}, also relative survival estimates 
are supplied in addition to observed survival figures. 

\code{surv.type = 'cif.obs'} requests cumulative incidence functions (CIF) 
to be estimated. 
CIFs are estimated for each competing risk based 
on a survival-interval-specific proportional hazards
assumption as described by Chiang (1968).  
With \code{surv.type = 'cif.rel'}, a CIF is estimated with using 
excess cases as the ''cause-specific'' cases. Finally, with 
\code{surv.type = 'surv.cause'}, cause-specific survivals are 
estimated separately for each separate type of event. 

In hazard-based estimation (\code{surv.method = "hazard"}) survival
time functions are transformations of the estimated corresponding hazard
in the intervals. The hazard itself is estimated using counts of events
(or excess events) and total subject-time in the interval. Life table
\code{surv.method = "lifetable"} estimates are constructed as transformations 
of probabilities computed using counts of events and counts of subjects 
at risk.


The vignette \href{../doc/survtab_examples.html}{survtab_examples} 
has some practical examples.
}

\section{Relative survival}{

 
When \code{surv.type = 'surv.rel'}, the user can choose 
\code{relsurv.method = 'pp'}, whereupon Pohar-Perme weighting is used.
By default \code{relsurv.method = 'e2'}, i.e. the Ederer II method
is used to estimate relative survival.
}

\section{Adjusted estimates}{


Adjusted estimates in this context mean computing estimates separately
by the levels of adjusting variables and returning weighted averages
of the estimates. For example, computing estimates separately by
age groups and returning a weighted average estimate (age-adjusted estimate).

Adjusting requires specification of both the adjusting variables and
the weights for all the levels of the adjusting variables. The former can be
accomplished by using \code{adjust()} with the argument \code{formula},
or by supplying variables directly to argument \code{adjust}. E.g. the
following are all equivalent:

\code{formula = fot ~ sex + adjust(agegr) + adjust(area)}

\code{formula = fot ~ sex + adjust(agegr, area)}

\code{formula  = fot ~ sex, adjust = c("agegr", "area")}

\code{formula  = fot ~ sex, adjust = list(agegr, area)}

The adjusting variables must match with the variable names in the
argument \code{weights};
see the \link[=direct_standardization]{dedicated help page}. 
Typically weights are supplied as a \code{list} or
a \code{data.frame}. The former can be done by e.g.

\code{weights = list(agegr = VEC1, area = VEC2)},

where \code{VEC1} and \code{VEC2} are vectors of weights (which do not
have to add up to one). See 
\href{../doc/survtab_examples.html}{survtab_examples} 
for an example of using a \code{data.frame} to pass weights.
}

\section{Period analysis and other data selection schemes}{


To calculate e.g. period analysis (delayed entry) estimates, 
limit the data when/before supplying to this function.See 
\href{../doc/survtab_examples.html}{survtab_examples}.
}

\examples{
\dontrun{
data("sire", package = "popEpi")
library(Epi)
library(survival)

## NOTE: recommended to use factor status variable
x <- Lexis(entry = list(FUT = 0, AGE = dg_age, CAL = get.yrs(dg_date)), 
           exit = list(CAL = get.yrs(ex_date)), 
           data = sire[sire$dg_date < sire$ex_date, ],
           exit.status = factor(status, levels = 0:2, 
                                labels = c("alive", "canD", "othD")), 
           merge = TRUE)

## phony group variable
set.seed(1L)
x$group <- rbinom(nrow(x), 1, 0.5)

## observed survival. explicit supplying of status:
st <- survtab(Surv(time = FUT, event = lex.Xst) ~ group, data = x, 
              surv.type = "surv.obs",
              breaks = list(FUT = seq(0, 5, 1/12)))
## this assumes the status is lex.Xst (right 99.9 \% of the time)
st <- survtab(FUT ~ group, data = x, 
              surv.type = "surv.obs",
              breaks = list(FUT = seq(0, 5, 1/12)))
              
## relative survival (ederer II)
data("popmort", package = "popEpi")
pm <- data.frame(popmort)
names(pm) <- c("sex", "CAL", "AGE", "haz")
st <- survtab(FUT ~ group, data = x, 
              surv.type = "surv.rel",
              pophaz = pm,
              breaks = list(FUT = seq(0, 5, 1/12)))

## ICSS weights usage
data("ICSS", package = "popEpi")
cut <- c(0, 30, 50, 70, Inf)
agegr <- cut(ICSS$age, cut, right = FALSE)
w <- aggregate(ICSS1~agegr, data = ICSS, FUN = sum)
x$agegr <- cut(x$dg_age, cut, right = FALSE)
st <- survtab(FUT ~ group + adjust(agegr), data = x, 
              surv.type = "surv.rel",
              pophaz = pm, weights = w$ICSS1,
              breaks = list(FUT = seq(0, 5, 1/12)))

#### using dates with survtab
x <- Lexis(entry = list(FUT = 0L, AGE = dg_date-bi_date, CAL = dg_date),
           exit = list(CAL = ex_date),
           data = sire[sire$dg_date < sire$ex_date, ],
           exit.status = factor(status, levels = 0:2, 
                                labels = c("alive", "canD", "othD")), 
           merge = TRUE)
## phony group variable
set.seed(1L)
x$group <- rbinom(nrow(x), 1, 0.5)

st <- survtab(Surv(time = FUT, event = lex.Xst) ~ group, data = x, 
              surv.type = "surv.obs",
              breaks = list(FUT = seq(0, 5, 1/12)*365.25))    
                  
## NOTE: population hazard should be reported at the same scale
## as time variables in your Lexis data.
data(popmort, package = "popEpi")
pm <- data.frame(popmort)
names(pm) <- c("sex", "CAL", "AGE", "haz")
## from year to day level
pm$haz <- pm$haz/365.25 
pm$CAL <- as.Date(paste0(pm$CAL, "-01-01")) 
pm$AGE <- pm$AGE*365.25 

st <- survtab(Surv(time = FUT, event = lex.Xst) ~ group, data = x, 
              surv.type = "surv.rel", relsurv.method = "e2",
              pophaz = pm,
              breaks = list(FUT = seq(0, 5, 1/12)*365.25))  
}
}
\references{
Perme, Maja Pohar, Janez Stare, and Jacques Estève. 
"On estimation in relative survival." Biometrics 68.1 (2012): 113-120.

Hakulinen, Timo, Karri Seppa, and Paul C. Lambert. 
"Choosing the relative survival method for cancer survival estimation." 
European Journal of Cancer 47.14 (2011): 2202-2210.
 
Seppa, Karri, Timo Hakulinen, and Arun Pokhrel. 
"Choosing the net survival method for cancer survival estimation." 
European Journal of Cancer (2013).

CHIANG, Chin Long. Introduction to stochastic processes in biostatistics. 
1968.

Seppa K., Dyba T. and Hakulinen T.: Cancer Survival, 
Reference Module in Biomedical Sciences. Elsevier. 08-Jan-2015 
doi: 10.1016/B978-0-12-801238-3.02745-8.
}
\seealso{
\code{\link{splitMulti}}, \code{\link{lexpand}}, 
\code{\link{ICSS}}, \code{\link{sire}}
\href{../doc/survtab_examples.html}{The survtab_examples vignette}

Other main functions: \code{\link{rate}},
  \code{\link{relpois_ag}}, \code{\link{relpois}},
  \code{\link{sirspline}}, \code{\link{sir}},
  \code{\link{survmean}}, \code{\link{survtab_ag}}

Other survtab functions: \code{\link{lines.survtab}},
  \code{\link{plot.survtab}}, \code{\link{print.survtab}},
  \code{\link{summary.survtab}}, \code{\link{survtab_ag}}
}
