% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rinit_spec.R
\name{rinit_spec}
\alias{rinit_spec}
\title{The initial-state distribution}
\description{
Specification of rinit
}
\details{
To fully specify the unobserved Markov state process, one must give its distribution at the zero-time (\code{t0}).
One does this by furnishing a value for the \code{rinit} argument.
As usual, this can be provided either as a C snippet or as an \R function.
In the former case, bear in mind that:
\enumerate{
  \item The goal of a this snippet is the construction of a state vector, i.e., the setting of the dynamical states at time \eqn{t_0}{t0}.
  \item In addition to the parameters and covariates (if any), the variable \code{t}, containing the zero-time, will be defined in the context in which the snippet is executed.
  \item \strong{NB:} The \code{statenames} argument plays a particularly important role when the rinit is specified using a C snippet.
   In particular, every state variable must be named in \code{statenames}.
   \strong{Failure to follow this rule will result in undefined behavior.}
 }
\link[=Csnippet]{General rules for writing C snippets can be found here}.

If an \R function is to be used, pass
\preformatted{
   rinit = f
}
to \code{pomp}, where \code{f} is a function with arguments that can include the initial time \code{t0}, any of the model parameters, and any covariates.
As usual, \code{f} may take additional arguments, provided these are passed along with it in the call to \code{pomp}.
\code{f} must return a named numeric vector of initial states.
It is of course important that the names of the states match the expectations of the other basic components.

Note that the state-process \code{rinit} can be either deterministic (as in the default) or stochastic.
In the latter case, it samples from the distribution of the state process at the zero-time, \code{t0}.
}
\section{Default behavior}{

By default, \code{pomp} assumes that the initial distribution is concentrated on a single point.
In particular, any parameters in \code{params}, the names of which end in \dQuote{\code{_0}} or \dQuote{\code{.0}}, are assumed to be initial values of states.
When the state process is initialized, these are simply copied over as initial conditions.
The names of the resulting state variables are obtained by dropping the suffix.
}

\section{Note for Windows users}{


Some Windows users report problems when using C snippets in parallel computations.
These appear to arise when the temporary files created during the C snippet compilation process are not handled properly by the operating system.
To circumvent this problem, use the \code{cdir} and \code{cfile} options \link[=pomp]{(described here)} to cause the C snippets to be written to a file of your choice, thus avoiding the use of temporary files altogether.
}

\examples{
## We set up a trivial process model:

trivial <- function (X, Y, ...) {
  c(X = X+1, Y = Y-1)
}

## We specify \code{rinit} with a function that
## sets state variables X and Y to the values in
## parameters X0, Y0:

f <- function (X0, Y0, ...) {
  c(X = X0, Y = Y0)
}

plot(simulate(times=1:5,t0=0,params=c(X0=3,Y0=-7),
  rinit=f,rprocess=onestep(trivial)))

## A function that depends on covariate P and
## time t0, as well as parameter X0:

g <- function (t0, X0, P, ...) {
  c(X = X0, Y = P + sin(2*pi*t0))
}

plot(simulate(times=1:5,t0=0,params=c(X0=3,Y0=-7),
  covar=covariate_table(t=0:10,P=3:13,times="t"),
  rinit=g,rprocess=onestep(trivial)))
}
\seealso{
More on implementing POMP models: 
\code{\link{Csnippet}},
\code{\link{accumulators}},
\code{\link{basic_components}},
\code{\link{covariate_table}()},
\code{\link{distributions}},
\code{\link{dmeasure_spec}},
\code{\link{dprocess_spec}},
\code{\link{parameter_trans}()},
\code{\link{pomp-package}},
\code{\link{prior_spec}},
\code{\link{rmeasure_spec}},
\code{\link{rprocess_spec}},
\code{\link{skeleton_spec}},
\code{\link{transformations}},
\code{\link{userdata}}
}
\concept{implementation_info}
