\name{nominal}
\alias{nominal}
\alias{print.nominal}
\alias{summary.nominal}
\alias{print.summary.nominal}

\title{
  Univariate and bivariate statistics for categorical,
  unordered variables
}
\description{

  \code{nominal} takes a data frame with categorical (i.e. nominal)
  variables and calculates a range of categorical statistics and
  posthoc analyses.

}
\usage{
nominal(formula, data, sort.bivariate = NULL, std.pearson.residual.min = 2,
   correct = FALSE, report.interval = 100, factor2logical = FALSE)

\method{print}{nominal}(x, max.print = 10,
   posthoc = "std.pearson.residuals.sign",
   assoc = ifelse("univariate" \%in\% class(x),
     list(c("N", "alpha.X2", "uc.12", "uc.21")),
     list(c("N1", "N2", "N12", "uc.12", "uc.21"))),
   sort.key = NULL, \dots)

\method{summary}{nominal}(object, posthoc = "std.pearson.residuals.sign",
   assoc = ifelse("univariate" \%in\% class(object),
     list(c("N", "alpha.X2", "uc.12", "uc.21")),
     list(c("N1", "N2", "N12", "uc.12", "uc.21"))),
   sort.key = NULL, \dots)

\method{print}{summary.nominal}(x, max.print = 10, \dots)

}

\arguments{
  \item{formula}{

  a formula (see \code{Details} specifying either (1) univariate
  analysis focusing on the relationship of one dependent categorical
  outcome variable (with possible multiple classes) and one or more
  categorical explanatory independent variables or (2) bivariate
  analysis scrutinizing the interrelationships of two or more
  categorical independent variables.

  }
  \item{data}{

  data frame (or an object coercible by \code{as.data.frame} to a
  data frame) containing the variables specified in the \code{formula}
  as columns. In the case of univariate analysis, the dependent
  outcome variable and the corresponding column can either be a
  multiple-value factor or a set of logical/binary variables. For both
  univariate and bivariate analyses, if
  \code{factor2logical=FALSE} (default), all independent
  variables and the respective columns in the data must be
  logical/binary; if \code{factor2logical=TRUE}, the variables
  can be factors which will then be automatically transformed into
  corresponding new logical/binary variables using the
  \code{\link{multinomial2logical}} function.

  }
  \item{x}{

  an object of class \code{"nominal"}, usually resulting from a call
  to \code{\link{nominal}}; or an object of class
  \code{"summary.nominal"}, usually resulting from a call to
  \code{\link{summary.nominal}}.

  }
  \item{object}{

  an object of class \code{"nominal"}, usually resulting from a call
  to \code{\link{nominal}}.

  }
  \item{sort.bivariate}{

  a character string, any one in the set
  \code{c("uc","lambda","tau")} specifying one three asymmetric
  measures of association (see \code{\link{associations}}), or
  \code{NULL}, specifying in the case of bivariate analysis whether
  the two contrasted categorical variables with be sorted so that the
  occurrence of the first categorical (logical/binary) variable in
  question (\code{category1} in the output) explains more of the
  occurrence (or absence) of the second categorical variable
  (\code{category2}), in contrast to the other way around. If
  e.g. \code{sort.bivariate="uc"}, the \code{uc.21>uc.12}, i.e.
  \eqn{UC(2|1)>UC(1|2)}. By default set to \code{NULL} so that no
  sorting will take place.

  }
  \item{std.pearson.residual.min}{

  the minimum absolute value for considering in univariate analysis a
  cellwise standardized Pearson residual in the contingency table to
  deviate significantly from the expected value (representing a
  homogeneous distribution); by default equal to 2 (see
  \code{\link{chisq.posthoc}}).

  }
  \item{correct}{

  a logical control parameter indicating whether to apply Yates'
  continuity correction when computing the (\eqn{X^2}) statistic in
  the chi-squared test (cf. \code{\link{chisq.posthoc}} and
  \code{\link{chisq.test}}); by default set to \code{FALSE}.

  }
  \item{report.interval}{

  a numeric variable indicating the interval at which the progress of
  the calculation of the statistics will be reported, by default set
  to \code{=100}. This is useful when the number of explanatory
  independent variables is great in the case of univariate analysis,
  and especially when their pairwise combinations are high in
  bivariate analysis, which is often the case.

  }
  \item{posthoc}{

  a character string (or vector of strings) specifying which of the
  posthoc analyses of a chi-squared test (any one of \code{c("X2",
  "X2.df.sign", "X2.df1.sign", "std.pearson.residuals",
  "std.pearson.residuals.sign")}) output by
  \code{\link{chisq.posthoc}}) will be output and included in
  \code{sumry.table} that is generated by \code{\link{print.nominal}}
  or \code{\link{summary.nominal}}. By default set to
  \code{"std.pearson.residuals.sign"}; if set to \code{NULL}, all
  posthoc analyses are excluded.

  }
  \item{assoc}{

  A character vector indicating which of the measures of association
  output by \code{\link{associations}} will be output and included in
  \code{sumry.table} generated by \code{\link{print.nominal}} or
  \code{\link{summary.nominal}}. By default set to \code{c("N",
  "alpha.X2", "uc.12", "uc.21")} in univariate analysis, and to
  \code{c("N1", "N2", "N12", "uc.12", "uc.21")} in bivariate
  analysis. If set to \code{NULL}, all measures of association are
  excluded.

  }
  \item{sort.key}{

  A character string specifying the measure of association, or some
  other statistic output by \code{nominal}, according to which
  the results will be sorted in decreasing order (any one of
  \code{c("uc.12", "uc.21", "lambda.12", "lambda.21", "tau.12",
  "tau.21")} for both univariate and bivariate analyses, and also
  \code{c("N", "alpha.X2")} for univariate analysis, and \code{c("N1",
  "N2", "N12")} for bivariate analysis). By default set to \code{NULL},
  in which case no sorting is done.

  }
  \item{factor2logical}{

  a logical indicating whether (multiple-value) factors in \code{data}
  referred to in \code{formula} should be automatically transformed into
  a corresponding set of binary/logical variables to be used in the
  ensuing analysis; by default set to \code{=FALSE}, in which case
  reference to non-logical variables will result in an error.

  }
  \item{max.print}{

  the maximum number of rows of the \code{parameter} to be output when
  printing with \code{print.summary.polytomous}; by default set to 20;
  if set to \code{NA} all rows will be output.

  }
  \item{\dots}{

  further arguments passed to or from other methods.

  }
}
\value{
For univariate analysis, a list with the following components:
\describe{

   \item{\code{univariate}}{

   a named list for each independent variable value (see
   \code{independents} below), containing for each a list with two
   elements, (1) \code{posthoc} with the results of
   \code{\link{chisq.posthoc}} and (2) \code{assoc} with the results of
   \code{\link{associations}}, both for the crosstabulation of the
   independent variable in question and dependent variable values

   }
   \item{\code{std.pearson.residuals.sign}}{

   a matrix of the cellwise assessments, for each independent variable
   (rows) per each dependent outcome variable value (columns), of
   whether the absolute value of a standardized Pearson residual is
   greater than \code{std.pearson.residual.min} (when the independent
   variable is present, i.e. \code{TRUE}), having the values
   \sQuote{+}, \sQuote{-}, or \sQuote{0}.

   }
   \item{\code{std.pearson.residuals}}{

   a matrix of cellwise standardized Pearson residuals, for each
   independent variable (rows) per each dependent outcome variable
   value (columns).

   }
   \item{\code{X2.df.sign}}{

   a matrix of the cellwise assessments, for each independent variable
   (rows) per each dependent outcome variable value (columns), of
   whether the chi-squared contribution of an individual cell by
   itself exceeds the overall minimum chi-squared value, based on the
   degrees of freedom for the full dimensions of the crosstabulation
   (cf. \code{X2.df} in \code{\link{chisq.posthoc}}), for a
   significantly non-homogeneous distribution of counts; having the
   values \sQuote{+} or \sQuote{-} when this is the case, with the
   sign indicating whether the cellwise observed value is greater or
   less than the expected value, or \sQuote{0} otherwise (indicating
   no significant divergence)

   }
   \item{\code{X2.df1.sign}}{

   a matrix of the cellwise assessments, for each independent variable
   (rows) per each dependent outcome variable value (columns), of
   whether the chi-squared contribution of an individual cell by
   itself exceeds the minimum chi-squared value, based on degrees of
   freedom as \code{=1} (cf. \code{X2.df1} in
   \code{\link{chisq.posthoc}}), for a significantly non-homogeneous
   distribution of counts; having the values \sQuote{+} or \sQuote{-}
   when this is the case, with the sign indicating whether the
   cellwise observed value is greater or less than the expected value,
   or \sQuote{0} otherwise (indicating no significant divergence)

   }
   \item{\code{X2}}{

   a matrix of the cellwise contributions, for each independent
   variable (rows) per each dependent outcome variable (columns), to
   the chi-squared statistic for each outcome variable value (columns)
   per each independent variable (rows), with the sign indicating
   whether the observed value is greater or less than the expected
   value.

   }
   \item{\code{assocs}}{

   a named list for each independent variable with the various
   measures of association calculated with \code{\link{associations}}
   between the independent variable in question and the values of the
   dependent outcome variable.

   In the notation of the association measures, the number codes in
   e.g. \code{uc.12} and \code{uc.21} refer to the direction
   (conditionality) of the asymmetric measures. For instance,
   \code{uc.12} corresponds to \eqn{UC(independent|dependent)},
   i.e. the reduction of uncertainty concerning the \code{independent}
   variable when knowing the dependent \code{outcome} variable,
   whereas \code{uc.21} corresponds to
   \eqn{UC(dependent|independent)}, i.e. the reduction of uncertainty
   concerning the \code{dependent} outcome variable when knowing the
   \code{independent} predictor variable.

   }
   \item{\code{dependents}}{

   a character string with the name of the dependent variable.

   }
   \item{\code{dependent.values}}{

   a character string vector with the multiple values, i.e. classes,
   of the dependent variable.

   }
   \item{\code{independents}}{

   a character string vector with the names of the independent
   variables.

   }
}

For bivariate analysis, a list containing the following components:
\describe{

   \item{\code{bivariate}}{

   a data frame with the various measures of association, calculated
   with \code{\link{associations}}, between the pairings of the
   independent variables (categories).

   The two independent variables to be contrasted, \code{category1}
   and \code{category2}, are provided in columns 1-2, the frequency of
   the first category \code{N1}, the frequency of the second category
   \code{N2}, and their joint frequency \code{N12}, in columns 3-5,
   followed by the rest of the association measures.

   In the notation of the association measures, the number codes in
   e.g. \code{uc.12} and \code{uc.21} refer to the direction
   (conditionality) of the asymmetric measures. For instance,
   \code{uc.12} corresponds to \eqn{UC(category1|category2)}, i.e. the
   reduction of uncertainty concerning \code{category1} when knowing
   \code{category2}, whereas \code{uc.21} corresponds to
   \eqn{UC(category2|category1)}, i.e. the reduction of uncertainty
   concerning \code{category2} when knowing \code{category1}.

   }
   \item{\code{independents}}{

   a character string vector with the names of the independent
   variables.

   }   
}

For \code{summary.nominal}, the list of results is supplemented with
one additional element:
\describe{

   \item{\code{sumry.table}}{

   a data frame with various statistics selected out of the results of
   \code{nominal} according to the arguments \code{posthoc} and
   \code{assoc} provided to \code{summary.nominal} (see above).

   For univariate analysis, each row consists of the statistics for
   each indepedent variable (in relation to the values of the
   dependent outcome variable values which are given as columns by
   default). For bivariate analysis, each row consists of the
   statistics for a (unique) pairing of two independent variables.

   }
}
}

\details{

The specification of the \code{formula} determines whether a
univariate or bivariate analysis is calculated. In univariate
analysis, one of the variables, e.g. an outcome, is considered
dependent of each of a number of other, independent variables,
i.e. explanatory predictors. In bivariate analysis no such single
dependent variable is assumed; in contrast, the interest is in the
pairwise interrelationships of the selected variables. In such a case,
each variable in these pairings is in both a dependent and independent
role.

Typical usages for univariate analysis are:
\preformatted{
nominal(formula=dependent ~ independent1 + independent2 + \dots, data=\dots)
nominal(formula=dependent1 + dependent2 + \dots ~ independent1 + \dots, data=\dots) 
nominal(formula=dependent1 + dependent2 + \dots ~ ., data=\dots) 
nominal(formula=dependent ~ ., data=\dots)
}

Typical usages for bivariate analysis are:
\preformatted{
nominal(formula=. ~ independent1 + independent2 + \dots, data=\dots)
nominal(formula=. ~ ., data=\dots)
}

In the univariate case, the formula has the form
\sQuote{\code{dependent ~ independent1 + independent2 + ...}}. Here,
the dependent variable is assumed to be a multiple-value factor and
the independent variables as binary/logical ones in \code{data},
Alternatively, the dependent variable can also be represented as
multiple binary/logical variables, so that formula takes the form
\sQuote{\code{dependent1 + dependent2 + ... ~ independent1 + ...}}. In
such a case, the values of the specified dependent variable values
must cover the entire data. In the special case
\sQuote{\code{dependent ~ .}}, the \sQuote{.} is taken to represent all
the other variables in \code{data} except the dependent one.

In the bivariate case, the formula has the form \sQuote{\code{. ~
independent1 + independent2 + ...}}. Here, the specified independent
variables are assumed to be logical/binary variables and are each
contrasted against all the others pairwise to study their
interrelationships. In the special case \sQuote{\code{. ~ .}}, all the
variables in the data are contrasted against each other
pairwise. N.B. in this latter case, too, all the variables are assumed
to be logical/binary ones.

}

\references{

  Agresti, A. (2002) Categorical Data Analysis (2nd
  edition). Hoboken: John Wiley & Sons, Hoboken.

  Arppe, A. (2008) Univariate, bivariate and multivariate methods in
  corpus-based lexicography -- a study of synonymy. Publications of
  the Department of General Linguistics, University of Helsinki,
  No. 44. URN: http://urn.fi/URN:ISBN:978-952-10-5175-3.

  Cohen, J. (1988) Statistical power analysis for the behavioral
  sciences, (2nd edition). Hillsdale: Lawrence Erlbaum Associates.

  Cramer, H. (1946) Mathematical Methods in Statistics. Princeton:
  Princeton University Press.

  Goodman, L. A. and W. H. Kruskal (1954) Measures of Association for
  Cross- Classifications. Journal of the American Statistical
  Association, Vol. 49, No.  268 (December 1954), pp. 732--764.

  Liebetrau, A. M. (1983) Measures of Association. Sage University
  Paper series on Quantitative Applications in the Social Sciences,
  07-032. Beverly Hills and London: Sage Publications.

  Theil, H. (1970) On the Estimation of Relationships Involving
  Qualitative Variables. The American Journal of Sociology, Vol. 76,
  No. 1 (July 1970), pp.  103--154.

}
\author{
Antti Arppe
}

\seealso{
See also \code{\link{chisq.posthoc}}, \code{\link{associations}}.
}

\examples{

data(think)
think.logical <- multinomial2logical(think, c("Agent","Patient"),
   outcome="Lexeme")

think.univariate <- nominal(Lexeme ~ ., data=think.logical)
summary(think.univariate)

think.bivariate <- nominal(. ~ ., data=think.logical[-1])
summary(think.bivariate)

}
\keyword{ univar category }
