% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_sundered_data.R
\name{get_sundered_data}
\alias{get_sundered_data}
\title{Sunder the data, splitting it into 'pass' and 'fail' pieces}
\usage{
get_sundered_data(
  agent,
  type = c("pass", "fail", "combined"),
  pass_fail = c("pass", "fail"),
  id_cols = NULL
)
}
\arguments{
\item{agent}{An agent object of class \code{ptblank_agent}. It should have had
\code{\link[=interrogate]{interrogate()}} called on it, such that the validation steps were actually
carried out.}

\item{type}{The desired piece of data resulting from the splitting. Options
for returning a single table are \code{"pass"} (the default) and \code{"fail"}. Each
of these options return a single table with, in the \code{"pass"} case, only the
rows that passed across all validation steps (i.e., had no failing test
units in any part of a row for any validation step), or, the complementary
set of rows in the \code{"fail"} case. Providing \code{NULL} returns both of the
split data tables in a list (with the names of \code{"pass"} and \code{"fail"}). The
option \code{"combined"} applies a categorical (pass/fail) label (settable in
the \code{pass_fail} argument) in a new \code{.pb_combined} flag column. For this
case the ordering of rows is fully retained from the input table.}

\item{pass_fail}{A vector for encoding the flag column with 'pass' and 'fail'
values when \code{type = "combined"}. The default is \code{c("pass", "fail")} but
other options could be \code{c(TRUE, FALSE)}, \code{c(1, 0)}, or \code{c(1L, 0L)}.}

\item{id_cols}{An optional specification of one or more identifying columns.
When taken together, we can count on this single column or grouping of
columns to distinguish rows. If the table undergoing validation is not a
data frame or tibble, then columns need to be specified for \code{id_cols}.}
}
\value{
A list of table objects if \code{type} is \code{NULL}, or, a single table if a
\code{type} is given.
}
\description{
Validation of the data is one thing but, sometimes, you want to use the best
part of the input dataset for something else. The \code{get_sundered_data()}
function works with an agent object that has intel (i.e., post
\code{interrogate()}) and gets either the 'pass' data piece (rows with no failing
test units across all row-based validation functions), or, the 'fail' data
piece (rows with at least one failing test unit across the same series of
validations). As a final option, we can have emit all the data with a new
column (called \code{.pb_combined}) which labels each row as passing or failing
across validation steps. These labels are \code{"pass"} and \code{"fail"} by default
but their values can be easily customized.
}
\details{
There are some caveats to sundering. The validation steps considered for this
splitting has to be of the row-based variety (e.g., the \verb{col_vals_*()}
functions or \code{\link[=conjointly]{conjointly()}}, but not \code{rows_distinct()}). Furthermore,
validation steps that experienced evaluation issues during interrogation are
not considered, and, validation steps where \code{active = FALSE} will be
disregarded. The collection of validation steps that fulfill the above
requirements for sundering are termed in-consideration validation steps.

If using any \code{preconditions} for validation steps, we must ensure that all
in-consideration validation steps use the same specified \code{preconditions}
function. Put another way, we cannot split the target table using a
collection of in-consideration validation steps that use different forms of
the input table.
}
\section{Function ID}{

8-3
}

\examples{
# Create a series of two validation
# steps focused on testing row values
# for part of the `small_table` object;
# `interrogate()` immediately
agent <-
  create_agent(
    read_fn = ~ small_table \%>\%
      dplyr::select(a:f),
    label = "`get_sundered_data()`"
  ) \%>\%
  col_vals_gt(vars(d), value = 1000) \%>\%
  col_vals_between(
    vars(c),
    left = vars(a), right = vars(d),
    na_pass = TRUE
  ) \%>\%
  interrogate()

# Get the sundered data piece that
# contains only rows that passed both
# validation steps (the default piece);
# this yields 5 of 13 total rows
agent \%>\% get_sundered_data()

# Get the complementary data piece:
# all of those rows that failed either
# of the two validation steps;
# this yields 8 of 13 total rows
agent \%>\%
  get_sundered_data(type = "fail")
  
# We can get all of the input data
# returned with a flag column (called
# `.pb_combined`); this is done by
# using `type = "combined"` and that
# rightmost column will contain `"pass"`
# and `"fail"` values
agent \%>\%
  get_sundered_data(type = "combined")

# We can change the `"pass"` or `"fail"`
# text values to another type of coding
# with the `pass_fail` argument; one
# possibility is TRUE/FALSE
agent \%>\%
  get_sundered_data(
    type = "combined",
    pass_fail = c(TRUE, FALSE)
  )

# ...and using `0` and `1` might be
# worthwhile in some situations
agent \%>\%
  get_sundered_data(
    type = "combined",
    pass_fail = 0:1
  )

}
\seealso{
Other Post-interrogation: 
\code{\link{all_passed}()},
\code{\link{get_agent_x_list}()},
\code{\link{get_data_extracts}()},
\code{\link{write_testthat_file}()}
}
\concept{Post-interrogation}
