% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/scape.R
\name{scape}
\alias{scape}
\title{Simulate phylogenetic community structure across a landscape}
\usage{
scape(tree, scape.size = 10, g.center = 1, g.range = 1, g.repulse = 1,
  wd.all = 150, signal.center = TRUE, signal.range = TRUE,
  same.range = TRUE, repulse = TRUE, center.scale = 1, range.scale = 1,
  repulse.scale = 1, site.stoch.scale = 0.5, sd.center = 1,
  sd.range = 1, rho = NULL, th = 8)
}
\arguments{
\item{tree}{\code{\link{phylo}} object}

\item{scape.size}{edge dimension of square landscape}

\item{g.center}{strength of phylogenetic signal in species range centers}

\item{g.range}{strength of phylogenetic signal in species range sizes}

\item{g.repulse}{strength of phylogenetic repulsion}

\item{wd.all}{niche width, larger values simulate broader range sizes}

\item{signal.center}{simulate with phylosignal in range centers}

\item{signal.range}{simulate with phylosignal in range size}

\item{same.range}{make all range sizes equal}

\item{repulse}{include phylogenetic repulsion in range centers}

\item{center.scale}{adjust strength of phylogenetic attraction in
range centers independent of \code{g.center}}

\item{range.scale}{adjust strength of phylogenetic signal in range
size independent of \code{g.range}}

\item{repulse.scale}{adjust strength of phylogenetic repulsion
independent of \code{g.repulse}}

\item{site.stoch.scale}{adjust strength of random variation in
species richness across sites}

\item{sd.center}{sd in \code{\link{rnorm}} for the range centers,
increase to get more variation in center values across species}

\item{sd.range}{sd \code{\link{rnorm}} for the range sizes,
increase to get more variation in range sizes across gradients}

\item{rho}{Grafen branch adjustment of phylogenetic tree see
\code{\link{corGrafen}}}

\item{th}{probability threshold 10^-th above which species are
considered present at a site}
}
\value{
\code{cc} \code{\link{comparative.comm}} object with
presence/absence results of simulations. The site names are the
row.columns of the cells in the original grid cells that made up
the data

\code{X.joint} full probabilities of species at sites, used
to construct cc

\code{X1} probabilities of species along gradient 1

\code{X2} probabilities of species along gradient 2

\code{sppXs} full probabilities of each species as an array
arranged in a \code{scape.size}-by-\code{scape.size} matrix

\code{V.phylo} initial phylogenetic covariance matrix from
tree

\code{V.phylo.rho} phylogenetic covariance matrix from tree
scaled by Grafen if rho is provided

\code{V.center} scaled by \code{g.center} phylo covariance
matrix used in the simulations

\code{V.range} scaled by \code{g.range} phylo covariance
matrix used in the simulations

\code{V.repulse} scaled by \code{g.repulse} phylo
covariance matrix used in the simulations

\code{bspp1} species optima for gradient 1

\code{bspp2} species optima for gradient 2

\code{u} the env gradients values for the two gradients

\code{wd} the denominator for species ranges
}
\description{
\code{scape} simulates communities that are phylogenetically structured
}
\details{
Simulates a landscape with species (i.e., tree tips)
distributions dependent on a supplied phylogenetic tree.  The
amount and type of structure is determened by the signal parameters
\code{g.center}, \code{g.range} and \code{g.repulse}. Parameters
are based on an Ornstein-Uhlenbeck model of evolution with
stabilizing selection. Values of g=1 indicate no stabilizing
selection and correspond to the Brownian motion model of evolution;
0<g<1 represents stabilizing selection; and g>1 corresponds to
disruptive selection where phylogenetic signal for the supplied
tree is amplified. See \code{\link{corBlomberg}}.  Communities are
simulated along two gradients where the positions along those
gradients, \code{g.center} and range sizes \code{g.range}, can
exhibit phylogenetic signal. Phylogenetic attraction is simulated
in the \code{g.center} paramter, while repulsion in
\code{g.repulse}. Both can be exhibited such that closly related
species are generally found with similar range centers
(phylogenetic attraction) but just not at the same site
(phylogenetic repulsion).  The function then returns probabilities
of of each species across sites and the presence and absences of
species based a supplied threshold, \code{th}, which can be
increased to obtain more species at sites and thus increase average
site species richness.
}
\examples{
#Create balanced tree with equal branch-lengths (signal in centers)
tree <- stree(8,type="balanced")
tree$edge.length <- rep(1, nrow(tree$edge))
tree$root <- 1
kk <- scape(tree, scape.size=100, g.center=100, g.range=1, g.repulse=1, wd.all=150,
    signal.center=TRUE, signal.range=FALSE, same.range=FALSE, repulse=FALSE,center.scale = 1,
    range.scale = 1, repulse.scale = 1, site.stoch.scale = 0, sd.center=3, sd.range=1,
    rho=NULL, th=20)

#Make some plots
par(mfrow=c(1,Ntip(tree)),mar=c(.1,.1,.1,.1))
for(j in seq_along(tree$tip.label))
    image(t(1 - kk$sppXs[,,j]/max(kk$sppXs[,,j])), xlab = "",
              ylab = "",main = "",axes=FALSE, col=grey.colors(10))

par(mfrow=c(2,1))
matplot((kk$X1), type = "l", xlab="gradient",ylab = "probability",
main = "Gradient 1",col=rainbow(dim(kk$X1)[2]),lty=1)
matplot((kk$X2), type = "l", xlab="gradient",ylab = "probability",
main = "Gradient 2",col=rainbow(dim(kk$X2)[2]),lty=1)

plot(x=seq_along(sites(kk$cc)),y = rowSums(comm(kk$cc)), main = "SR",type = "l")
cor(kk$X1)
}
\author{
M.R. Helmus, cosmetic changes by Will Pearse
}
\references{
Helmus M.R. & Ives A.R. (2012). Phylogenetic diversity
area curves. Ecology, 93, S31-S43.
}
\seealso{
\code{\link{eco.scape}} \code{\link{sim.phy}}
\code{\link{sim.meta}}
}

