% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/twoWayEcologyCluster.R
\name{twoWayEcologyCluster}
\alias{twoWayEcologyCluster}
\title{R-Mode vs Q-Mode Two-Way Cluster Analyses and Abundance Plot for Community Ecology Data}
\usage{
twoWayEcologyCluster(xDist, yDist, propAbund, clustMethod = "average",
  marginBetween = 0.1, abundExpansion = 4, xAxisLabel = "Across Sites",
  yAxisLabel = "Across Taxa")
}
\arguments{
\item{xDist}{The pair-wise distance matrix for the cluster diagram drawn along the
horizontal axis of the graphic. Should be a distance matrix, or a matrix that can
be coerced to a distance matrix, for the same number of units as rows in \code{propAbund}.}

\item{yDist}{The pair-wise distance matrix for the cluster diagram drawn along the
vertical axis of the graphic. Should be a distance matrix, or a matrix that can
be coerced to a distance matrix, for the same number of units as columns in \code{propAbund}.}

\item{propAbund}{A matrix of abundance data, preferably relative
abundance scaled as proportions of the total number of individuals
at each site. This data determines the size scale of the taxon/site dots.}

\item{clustMethod}{The agglomerative clustering method used, as with
argument \code{method} with function \code{hclust}. \code{clustMethod} must
be one of \code{"average"} (the default method for this function, also known as
average-linkage or as UPGMA), \code{"ward.D"}, \code{"ward.D2"}, \code{"single"}, \code{"complete"},
\code{"mcquitty"} (also known as WPGMA), \code{"median"}
(also known as WPGMC) or \code{"centroid"} (also known as UPGMC).}

\item{marginBetween}{Argument controlling space placed between
the cluster diagrams and the abundance plot. Default is 0.1.}

\item{abundExpansion}{An argument that is a multiplier controlling the size
of dots plotted for reflecting relative abundance.}

\item{xAxisLabel}{The label placed on the horizontal axis of the plot.}

\item{yAxisLabel}{The label placed on the vertical axis of the plot.}
}
\value{
This function creates a plot, and returns nothing, not even invisible output.
}
\description{
This mode plots both R-mode (across sites) and Q-mode (across taxa) dendrograms
for a community ecology dataset, with branches aligned with a grid of dots
representing the relative abundance of taxa at each site in the dataset.
}
\details{
You might be able to apply this to datasets that aren't community ecology datasets
of proportional abundance, but I can't guarantee pr even predict what will happen.
}
\examples{
\dontrun{
library(vegan)

# generate random community ecology data
	# using a Poisson distribution
data<-matrix(rpois(5*7,1),5,7)

# relative abundance, distance matrices
propAbundMat<-t(apply(data,1,function(x) x/sum(x)))
rownames(propAbundMat)<-paste0("x", 1:nrow(propAbundMat))
colnames(propAbundMat)<-paste0("y", 1:ncol(propAbundMat))

siteDist<-vegdist(propAbundMat, "bray")
taxaDist<-vegdist(t(propAbundMat), "bray")

dev.new(width=10)	

twoWayEcologyCluster(
	xDist = siteDist, 
	yDist = taxaDist,
	propAbund = propAbundMat
	)

##############################################	

# now let's try an example with some pokemon data
data(kanto)

# get distance matrices for sites and taxa
	# based on bray-curtis dist
	# standardized to total abundance

# standardize site matrix to relative abundance
siteStand <- decostand(kanto, method = "total")

# calculate site distance matrix (Bray-Curtis)
siteDist <- vegdist(siteStand, "bray")

# calculate taxa distance matrix (Bray-Curtis)
	# from transposed standardized site matrix 
taxaDist <- vegdist(t(siteStand), "bray")

dev.new(width=10)	

twoWayEcologyCluster(
    xDist = siteDist,
    yDist = taxaDist,
    propAbund = siteStand
    )
}
}
\references{
The function here was designed to emulate previous published 'two-way'
cluster diagrams, particularly the one in Miller, 1988:

Miller, A. I. 1988. Spatial Resolution in Subfossil Molluscan
Remains: Implications for Paleobiological Analyses. \emph{Paleobiology} 14(1):91-103.
}
\seealso{
Several other functions for community ecology data in paleotree are described at the \code{\link{communityEcology}} help file. Also see the example dataset, \code{\link{kanto}}.
}
\author{
David W. Bapst
}
