\name{expandTaxonTree}
\alias{expandTaxonTree}
\title{Extrapolating Lower-Level Taxon Phylogenies from Higher-Level Taxon Trees}
\description{
This function takes a tree composed of higher-leve taxa and a vector of lower-level taxa belonging to the set of higher-level taxa included in the input tree and produces a tree composed of the lower-level taxa, by treating the higher-level taxa as unresolved monophyletic polytomies. A user can also mark higher taxa as paraphyletic such that these are secondarily collapsed and do not form monophyletic clades in the output tree.
}
\usage{
expandTaxonTree(taxonTree, taxaData, collapse = NULL, plot = FALSE)
}
\arguments{
  \item{taxonTree}{A phylo object where tips represent higher taxa}
  \item{taxaData}{Character vector of higher taxa, with elements names equal to the lower taxa. See below.}
  \item{collapse}{Character vector of non-monophyletic higher taxa to be collapsed}
  \item{plot}{If true, plots a comparison between input and output trees}
}
\details{
The output tree will probably be a rough unresolved view of the relationships among the taxa, due to the treatment of higher-level taxa as polytomies. This is similar to the methods used in Webb and Donoghue (2005) and Friedman (2009). Any analyses should be done by resolving this tree with \code{\link{multi2di}} in the ape package or via the various time-scaling functions found in this package (paleotree).

The taxaData vector should have one element per lower-level taxon that is to be added to the tree. The name of each element in the vector should be the names of the lower-level taxa, which will be used as new tip labels of the output lower-taxon tree.

The collapse vector should be used if there is doubt about the monophyly of any of the higher-level taxa included in the input taxontree. By default, all higher-level taxa are treated as monophyletic if not otherwise specified.

The output tree will always lack branch lengths and thus will not be time-scaled.
}
\value{Outputs the modified tree as an object of class phylo, with the higher-level taxa expanded into polytomies and the lower-level taxa as the tip labels.}
\references{
Friedman, M. 2009 Ecomorphological selectivity among marine teleost fishes during the end-Cretaceous extinction. \emph{Proceedings of the National Academy of Sciences} \bold{106}(13):5218--5223.

Webb, C. O., and M. J. Donoghue. 2005 Phylomatic: tree assembly for applied phylogenetics. \emph{Molecular Ecology Notes} \bold{5}(1):181--183.
}
\author{David W. Bapst}
\seealso{
\code{\link{multi2di}}, \code{\link{bind.tree}}
}
\examples{
set.seed(444)
#lets make our hypothetical simulated tree of higher taxa
taxtr <- rtree(10)
taxd <- sample(taxtr$tip.label,30,replace=TRUE)	#taxa to place within higher taxa
names(taxd) <- paste(taxd,"_x",1:30,sep="")
coll <- sample(taxtr$tip.label,3)		#what to collapse?
expandTaxonTree(taxonTree=taxtr,taxaData=taxd,collapse=coll,plot=TRUE)
}