% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fun-wrap.r
\name{wrap-ord}
\alias{wrap-ord}
\alias{eigen_ord}
\alias{svd_ord}
\alias{cmdscale_ord}
\alias{cancor_ord}
\title{Wrappers for lossy ordination methods}
\usage{
eigen_ord(x, symmetric = isSymmetric.matrix(x))

svd_ord(x, nu = min(dim(x)), nv = min(dim(x)))

cmdscale_ord(d, k = 2, add = FALSE)

cancor_ord(x, y, xcenter = TRUE, ycenter = TRUE, scores = FALSE)
}
\arguments{
\item{x}{a numeric or complex matrix whose spectral decomposition is to
    be computed.  Logical matrices are coerced to numeric.}

\item{symmetric}{if \code{TRUE}, the matrix is assumed to be symmetric
    (or Hermitian if complex) and only its lower triangle (diagonal
    included) is used.  If \code{symmetric} is not specified,
    \code{\link[base]{isSymmetric}(x)} is used.}

\item{nu}{the number of left  singular vectors to be computed.
    This must between \code{0} and \code{n = nrow(x)}.}

\item{nv}{the number of right singular vectors to be computed.
    This must be between \code{0} and \code{p = ncol(x)}.}

\item{d}{a distance structure such as that returned by \code{dist}
    or a full symmetric matrix containing the dissimilarities.}

\item{k}{the maximum dimension of the space which the data are to be
    represented in; must be in \eqn{\{1, 2, \ldots, n-1\}}{{1, 2, \\dots, n-1}}.}

\item{add}{logical indicating if an additive constant \eqn{c*} should
    be computed, and added to the non-diagonal dissimilarities such that
    the modified dissimilarities are Euclidean.}

\item{y}{numeric matrix (\eqn{n \times p_2}{n * p2}), containing the
    y coordinates.}

\item{xcenter}{logical or numeric vector of length \eqn{p_1}{p1},
    describing any centering to be done on the x values before the
    analysis.  If \code{TRUE} (default), subtract the column means.
    If \code{FALSE}, do not adjust the columns.  Otherwise, a vector
    of values to be subtracted from the columns.}

\item{ycenter}{analogous to \code{xcenter}, but for the y values.}

\item{scores}{Logical; whether to return canonical scores and structure
correlations.}
}
\value{
Objects slightly modified from the outputs of the original functions,
with new '*-ord' classes.
}
\description{
These \verb{*_ord} functions wrap core R functions with modifications
for use with '\link{tbl_ord}' methods. Some parameters are hidden from the user
and set to settings required for these methods, some matrix outputs are
given row or column names to be used by them, and new '*_ord' S3 class
attributes are added to enable them.
}
\details{
The following table summarizes the wrapped functions:\tabular{lllll}{
   Original function \tab Hide params \tab New params \tab Add names \tab New class \cr
   \code{\link[base:eigen]{base::eigen()}} \tab Yes \tab No \tab Yes \tab Yes \cr
   \code{\link[base:svd]{base::svd()}} \tab Yes \tab No \tab Yes \tab Yes \cr
   \code{\link[stats:cmdscale]{stats::cmdscale()}} \tab Yes \tab No \tab No \tab Yes \cr
   \code{\link[stats:cancor]{stats::cancor()}} \tab No \tab Yes \tab No \tab Yes \cr
}


By default, \code{\link[=cancor_ord]{cancor_ord()}} returns the same data as \code{\link[stats:cancor]{stats::cancor()}}: the
canonical correlations (\code{cor}), the canonical coefficients (\verb{$xcoef} and
\verb{$ycoef}), and the variable means (\verb{$xcenter}, \verb{$ycenter}). If \code{scores = TRUE}, then \code{\link[=cancor_ord]{cancor_ord()}} also returns the scores \verb{$xscores} and \verb{$yscores}
calculated from the (appropriately centered) data and the coefficients and
the intraset structure correlations \verb{$xstructure} and \verb{$ystructure} between
these and the data. These modifications are inspired by the \code{cancor()}
function in \strong{candisc}, though two caveats should be noted: First, the
canonical coefficients (hence the canonical scores) are scaled by \eqn{n - 1}
compared to these, though the intraset structure correlations are the same.
Second, the \emph{interset} structure correlations are not returned, as these may
be obtained by conferring inertia unto the intraset ones.
}
\examples{
# glass composition data from one furnace
glass_banias <- subset(
  glass,
  Context == "L.15;B.166",
  select = c("SiO2", "Na2O", "CaO", "Al2O3", "MgO", "K2O")
)
# eigendecomposition of a covariance matrix
(glass_cov <- cov(glass_banias))
eigen_ord(glass_cov)
# singular value decomposition of a data matrix
svd_ord(glass_banias)
# classical multidimensional scaling of a distance matrix
cmdscale_ord(dist(glass_banias))

# canonical correlation analysis with trace components
glass_banias_minor <- subset(
  glass,
  Context == "L.15;B.166",
  select = c("TiO2", "FeO", "MnO", "P2O5", "Cl", "SO3")
)
# impute half of detection threshold
glass_banias_minor$TiO2[[1L]] <- 0.5
cancor_ord(glass_banias, glass_banias_minor)

# calculate canonical scores and structure correlations
glass_cca <-
  cancor_ord(glass_banias[, 1:3], glass_banias_minor[, 1:3], scores = TRUE)
# scores
glass_cca$xscores
# intraset correlations
glass_cca$xstructure
# interset correlations
glass_cca$xstructure \%*\% diag(glass_cca$cor)
}
