% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/replacement_costs.R
\name{replacement_costs}
\alias{replacement_costs}
\title{Replacement cost}
\usage{
replacement_costs(x, solution, n = 1)
}
\arguments{
\item{x}{project prioritization \code{\link[=problem]{problem()}}.}

\item{solution}{\code{\link[base:data.frame]{base::data.frame()}} or
\code{\link[tibble:tibble]{tibble::tibble()}} table containing the solutions. Here,
rows correspond to different solutions and columns correspond to
different actions. Each column in the argument to \code{solution} should
be named according to a different action in \code{x}.
Cell values indicate if an action is funded in a given solution or not,
and should be either zero or one. Arguments to \code{solution} can
contain additional columns, and they will be ignored.}

\item{n}{\code{integer} solution number to calculate replacement cost values.
Since each row in the argument to \code{solutions} corresponds to a
different solution, this argument should correspond to a row in
the argument to \code{solutions}. Defaults to 1.}
}
\value{
A \code{\link[tibble:tibble]{tibble::tibble()}} table containing the following
columns:

\describe{

\item{\code{"action"}}{\code{character} name of each action.}

\item{\code{"cost"}}{\code{numeric} cost of each solution when each
action is locked out.}

\item{\code{"obj"}}{\code{numeric} objective value of each solution when
each action is locked out. This is calculated using the objective
function defined for the argument to \code{x}.}

\item{\code{"rep_cost"}}{\code{numeric} replacement cost for each
action. Greater values indicate greater irreplaceability. Missing
(\code{NA}) values are assigned to actions which are not selected for
funding in the specified solution, infinite (\code{Inf}) values are
assigned to to actions which are required to meet feasibility
constraints, and negative values mean that superior solutions than
the specified solution exist.}

}
}
\description{
Calculate the replacement cost for priority actions in a project
prioritization \code{\link[=problem]{problem()}} (Moilanen \emph{et al.} 2009). Actions associated
with larger replacement cost values are more irreplaceable, and may
need to be implemented sooner than actions with lower replacement cost
values.
}
\details{
Replacement cost values are calculated for each priority action
specified in the solution. Missing (\code{NA}) values are assigned to
actions which are not selected for funding in the specified solution.
For a given action, its replacement cost is calculated by
(i) calculating the objective value for the optimal solution to
the argument to \code{x}, (ii) calculating the objective value for the
optimal solution to the argument to \code{x} with the given action locked
out, (iii) calculating the difference between the two objective
values, (iv) the problem has an objective which aims to minimize
the objective value (only \code{\link[=add_min_set_objective]{add_min_set_objective()}}, then
the resulting value is multiplied by minus one so that larger values
always indicate actions with greater irreplaceability. Please note this
function can take a long time to complete
for large problems since it involves re-solving the problem for every
action selected for funding.
}
\examples{
\dontrun{
# load data
data(sim_projects, sim_features, sim_actions)

# build problem with maximum richness objective and $400 budget
p <- problem(sim_projects, sim_actions, sim_features,
             "name", "success", "name", "cost", "name") \%>\%
     add_max_richness_objective(budget = 400) \%>\%
     add_feature_weights("weight") \%>\%
     add_binary_decisions()

# solve problem
s <- solve(p)

# print solution
print(s)

# calculate replacement cost values
r <- replacement_costs(p, s)

# print output
print(r)

# plot histogram of replacement costs,
# with this objective, greater values indicate greater irreplaceability
hist(r$rep_cost, xlab = "Replacement cost", main = "")
}
}
\references{
Moilanen A, Arponen A, Stokland JN & Cabeza M (2009) Assessing replacement
cost of conservation areas: how does habitat loss influence priorities?
\emph{Biological Conservation}, \strong{142}, 575--585.
}
\seealso{
\code{\link[=solution_statistics]{solution_statistics()}},
\code{\link[=project_cost_effectiveness]{project_cost_effectiveness()}}.
}
