% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/odeintr.R
\name{compile_sys}
\alias{compile_sys}
\title{Compile ODE system}
\usage{
compile_sys(name, sys, pars = NULL, const = FALSE, method = "rk5_i",
  sys_dim = -1L, atol = 1e-06, rtol = 1e-06, globals = "",
  headers = "", footers = "", compile = TRUE, observer = NULL,
  env = new.env(), ...)
}
\arguments{
\item{name}{the name of the generated integration function}

\item{sys}{a string containing C++ expressions}

\item{pars}{a named vector of numbers or a vector of names or number of parameters}

\item{const}{declare parameters const if true}

\item{method}{a method string (see Details)}

\item{sys_dim}{length of the state vector}

\item{atol}{absolute tolerance if using adaptive step size}

\item{rtol}{relative tolerance if using adaptive step size}

\item{globals}{a string with global C++ declarations}

\item{headers}{code to appear before the \code{odeintr} namespace}

\item{footers}{code to appear after the \code{odeintr} namespace}

\item{compile}{if false, just return the code}

\item{observer}{an optional R function to record output}

\item{env}{install functions into this environment}

\item{...}{passed to \code{\link{sourceCpp}}}
}
\value{
The C++ code invisibly.

The following functions are generated:

\tabular{llll}{
 Function \tab Use \tab Arguments \tab Return \cr
 \code{name} \tab
   regular observer calls \tab
   \code{init, duration, step_size = 1.0, start = 0.0} \tab
   data frame\cr
 \code{name_adap} \tab
   adaptive observer calls \tab
   \code{init, duration, step_size = 1.0, start = 0.0} \tab
   data frame \cr
 \code{name_at} \tab
   specified observer calls \tab
   \code{init, times, step_size = 1.0, start = 0.0} \tab
   data frame \cr
 \code{name_continue_at} \tab
   specified observer calls starting from previous final state \tab
   \code{times, step_size = 1.0} \tab
   data frame \cr
 \code{name_no_record} \tab
   no observer calls \tab
   \code{init, duration, step_size = 1.0, start = 0.0} \tab
   vector (final state)\cr
 \code{name_reset_observer} \tab
   clear observed record \tab void \tab void \cr
 \code{name_get_state} \tab
   get current state \tab void \tab vector \cr
 \code{name_set_state} \tab
   set current state \tab \code{new_state} \tab void \cr
 \code{name_get_output} \tab
   fetch observed record \tab void \tab data frame \cr
 \code{name_get_params} \tab
   get parameter values \tab void \tab a list \cr
 \code{name_set_params} \tab
   set parameter values \tab parameters \tab void}

Arguments are:

\tabular{ll}{
\code{init} \tab vector of initial conditions \cr
\code{duration} \tab end at start + duration \cr
\code{step_size} \tab the integration step size; variable for adaptive methods \cr
\code{start} \tab the starting time (always equal 0.0 for \code{name_at}) \cr
\code{time} \tab vector of times as which to call the observer \cr
\code{new_state} \tab vector of state values \cr}
}
\description{
Generates an integrator using Rcpp
}
\details{
C++ code is generated and compiled with
\code{\link{sourceCpp}}. The returned function will
integrate the system starting from a provided initial
condition and initial time to a specified final time.
An attempt is made to get the length of the state vector
from the system definition. If this fails, the code will
likely crash your R session. It is safer to specify
\code{sys_dim} directly.

The C++ expressions must index a state array of length
\code{sys_dim}. The state array is \code{x} and the
derivatives are \code{dxdt}. The first state value is
\code{x[0]} and the first derivative is \code{dxdt[0]}.

In the case you use bare \code{dxdt} and \code{x}, an
attempt will be made to append \code{[0]} to these
variables. This can fail, so do not rely on it.
This will also fail if you set \code{sys_dim}
to a positive value.

The \code{globals} string can be arbitrary C++ code. You
can set global named parameter values here. Note that
these will be defined within the \code{odeintr} namespace.

If you supply the \code{pars} argument, these parameters
will be compiled into the code. There are three options:
1) if \code{pars} is a single number, then you can access
a vector of parameters named \code{pars} of the specified
length; 2) if \code{pars} is a character vectors, then a
parameter will be defined for each; and 3) if the character
vector is named, then the names will be used for the
parameter names and the associated values will be used
as defaults. If you specify \code{const = TRUE}, these
named parameters will be declared const. Otherwise
parameter getter/setter functions will be defined.

If \code{observer} is a function, then this function will
be used to record the output of the integration. It is called
with signature \code{obsever(x, t)}. Its return value will
be coerced to a list. Observer getter/setter functions will be
emitted as well (\code{name_g(s)et_observer}). You can also
get and set an output processing function (\code{name_g(s)et_output_processor}).
It will be passed
a 2-element list. The first element is a vector of time points
and the 2nd element is a list of lists, one list per time
point. The default processor converts this to a data frame.

You can insert arbitrary code outside the \code{odeintr}
name space using \code{headers} and \code{footers}. This code
can be anything compatible with Rcpp. You could for example
define exported Rcpp functions that set simulation paramters.
\code{headers} is inserted right after the Rcpp and ODEINT
includes. \code{footers} is inserted at the end of the
code.

The following methods can be used:

\tabular{lll}{
Code \tab Stepper \tab Type \cr
\code{euler} \tab \code{euler} \tab Interpolating \cr
\code{rk4} \tab \code{runge_kutta4} \tab Regular \cr
\code{rk54} \tab \code{runge_kutta_cash_karp54} \tab Regular \cr
\code{rk54_a} \tab \code{runge_kutta_cash_karp54} \tab Adaptive \cr
\code{rk5} \tab \code{runge_kutta_dopri5} \tab Regular \cr
\code{rk5_a} \tab \code{runge_kutta_dopri5} \tab Adaptive \cr
\code{rk5_i} \tab \code{runge_kutta_dopri5} \tab Interpolating adaptive \cr
\code{rk78} \tab \code{runge_kutta_fehlberg78} \tab Regular \cr
\code{rk78_a} \tab \code{runge_kutta_fehlberg78} \tab Adaptive \cr
\code{abN} \tab \code{adams_bashforth} \tab Order N multistep \cr
\code{abmN} \tab \code{adams_bashforth_moulton} \tab Order N multistep \cr
\code{bs} \tab \code{bulirsch_stoer} \tab Adaptive \cr
\code{bsd} \tab \code{bulirsch_stoer_dense_out} \tab Interpolating adaptive}

These steppers are described at \href{http://headmyshoulder.github.io/odeint-v2/doc/boost_numeric_odeint/odeint_in_detail/steppers.html#boost_numeric_odeint.odeint_in_detail.steppers.stepper_overview}{here}.
}
\note{
The c++11 plugin is enabled.
}
\examples{
\dontrun{
# Logistic growth
compile_sys("logistic", "dxdt = x * (1 - x)")
plot(logistic(0.001, 15, 0.1), type = "l", lwd = 2, col = "steelblue")
Sys.sleep(0.5)

# Lotka-Volterra predator-prey equations
pars = c(alpha = 1, beta = 1, gamma = 1, delta = 1)
LV.sys = '
  dxdt[0] = alpha * x[0] - beta * x[0] * x[1];
  dxdt[1] = gamma * x[0] * x[1] - delta * x[1];
' # LV.sys
compile_sys("preypred", LV.sys, pars, TRUE)
x = preypred(rep(2, 2), 100, 0.01)
plot(x[, 2:3], type = "l", lwd = 2,
     xlab = "Prey", ylab = "Predator",
     col = "steelblue")
Sys.sleep(0.5)

# Lorenz model from odeint examples
pars = c(sigma = 10, R = 28, b = 8 / 3)
Lorenz.sys = '
  dxdt[0] = sigma * (x[1] - x[0]);
  dxdt[1] = R * x[0] - x[1] - x[0] * x[2];
  dxdt[2] = -b * x[2] + x[0] * x[1];
' # Lorenz.sys
compile_sys("lorenz", Lorenz.sys, pars, TRUE)
x = lorenz(rep(1, 3), 100, 0.001)
plot(x[, c(2, 4)], type = 'l', col = "steelblue")
}
}
\author{
Timothy H. Keitt
}
\seealso{
\code{\link{set_optimization}}, \code{\link{integrate_sys}}
}

