\name{read.ctd}
\alias{read.ctd}
\title{Read a CTD data file}
\description{
Read a CTD data file, producing an object of type \code{ctd}.
}
\usage{
read.ctd(file, type=NULL, debug=FALSE,
 	columns=NULL, check.human.headers=TRUE)
}
\arguments{
  \item{file}{a connection or a character string giving the name of the
    file to load.}
  \item{type}{if \code{NULL}, then the first line is studied, in order to determine the file type.  If \code{type="SBE19"}, then a \emph{Seabird 19} (or similar) CTD
format is assumed. If \code{type="WOCE"} then a 
WOCE-exchange file is assumed.}
  \item{debug}{a flag that can be set to \code{TRUE} to turn on debugging.}
  \item{columns}{if \code{NULL}, then \code{read.ctd} tries to infer column names from the
                 header.  If a list, then it will be taken to be the list
				 of columns.  The list must include \code{"pressure"}, \code{"temperature"}
				 and either \code{"conductivity"} or \code{"salinity"}, or else very little
				 can be done with the file.}
  \item{check.human.headers}{produces warnings for missing human-written header items.}
}
\details{
  Oceanographers use CTD (conductivity-temperature-depth) instruments
  to measure key properties of the ocean, such as water temperature,
  salinity, etc.  This function reads CTD datasets that have been stored
  in common formats, and could be extended to 
  accommodate other formats if needed.

  \code{read.ctd} does a reasonable job of inferring meta-information
  from file headers, but it still has limitations.  For
  example, in the first file tested during development, 
  the sampling rate was written as
  \code{* sample rate = 1 scan every 0.5 seconds}, while 
  in the second test file it was written 
  \code{* Real-Time Sample Interval = 0.125 seconds}.  Similarly,
  \code{read.ctd} can be challenged in parsing latitudes and longitudes
  in the wide variety of ways that humans choose.  Still, such limitations
  are not really pressing in practice, since the \code{ctd} object is made available
  for manipulation.  If \code{read.ctd} cannot scan \code{33 and a third} as a latitude,
  just examine the header (stored as a list in \code{object$header}), and do something
  like \code{object$latitude <- 33 + 1/3}.

  It should be noted that different file types provide different meta-information.
  For example, the WOCE exchange format binds together the institute name and the initials
  of the chief scientist into a single string that \code{read.ctd} cannot parse, so
  both \code{object$institute} and \code{object$scientist} are left blank for WOCE files.
}
\value{
  A \code{ctd} object containing information about the station
  (e.g. \code{latitude}, etc.), along with vectors containing the
  acquired data (e.g. \code{S}, etc.).  The full header is also
  contained in the list.  The returned list is as follows, but the reader
  is cautioned that some items may be blank for some file formats.
  \item{header[]}{the header itself, normally containing several dozen lines of information}
  \item{filename}{name of the file passed to read.ctd}
  \item{filename.orig}{name of the original file saved by the instrument (normally a hex file)}
  \item{system.upload.time}{system upload time}
  \item{ship}{name of the ship from which the CTD was deployed}
  \item{scientist}{name of the scientist taking the data}
  \item{institute}{name of the institute}
  \item{address}{the address of the institute where the scientist}
  \item{cruise}{name of cruise}
  \item{section.id}{name of section}
  \item{station}{station number or name}
  \item{date}{date of lowering of CTD into the water}
  \item{latitude}{latitude, in decimal degrees, positive north of equator}
  \item{longitude}{longitude, in decimal degrees, positive if east
    of Greenwich and west of dateline} 
  \item{recovery}{date of recovery of CTD}
  \item{sample.interval}{time interval between samples [s]}
  \item{water.depth}{the water depth at the site [m]}
  \item{processing.log}{A processing log.}
  \item{data}{A data table containing the profile data as vectors.  The column
    names are discovered from the header, and may thus differ from file
    to file.  For example, some CTD instruments may have a fluorometer
    connected, others may not.  The following vectors are normally
    present: \code{data$pressure}, 
    \code{data$salinity}, \code{data$temperature}, and \code{data$sigma}.
    Note that \code{data$sigma} is calculated 
    from the pressure, salinity, and temperature in the file, using 
    \code{\link{sw.sigma}}.}
}

\seealso{
The generic function \code{\link{read.oce}} provides an alternative to this.

A \code{ctd} object may be summarized with \code{\link{summary.ctd}} and plotted
with \code{\link{plot.ctd}}.
}
\examples{
\dontrun{
library(oce)
x <- read.ctd("/usr/local/lib/R/library/oce/demo/ctdprofile.cnv")
plot(x) # summary with TS and profiles
plot.TS(x) # just the TS
}
}
\references{
The Seabird CTD instrument is described at
\url{http://www.seabird.com/products/spec_sheets/19plusdata.htm},
and  
the WOCE-exchange format is described at
\url{http://www.nodc.noaa.gov/woce_V2/disk02/exchange/exchange_format_desc.htm}
}
\author{Dan Kelley \email{Dan.Kelley@Dal.Ca}}
\keyword{misc}
