% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nma_thresh.R
\name{nma_thresh}
\alias{nma_thresh}
\title{Calculate thresholds and invariant intervals}
\usage{
nma_thresh(mean.dk, lhood, post, nmatype = "fixed", X = NULL,
  mu.design = NULL, delta.design = NULL, opt.max = TRUE,
  trt.rank = 1, trt.code = NULL, trt.sub = NULL, mcid = 0,
  mcid.type = "decision")
}
\arguments{
\item{mean.dk}{Posterior means of basic treatment parameters \eqn{d_k}.}

\item{lhood}{Likelihood (data) covariance matrix.}

\item{post}{Posterior covariance matrix (see details).}

\item{nmatype}{Character string, giving the type of NMA performed. One of
"fixed" (fixed effects, the default) or "random" (random effects). May be
abbreviated.}

\item{X}{[FE models only] Design matrix for basic treatment parameters.}

\item{mu.design}{[RE models only] Design matrix for any extra parameters.
Defaults to \code{NULL} (no extra parameters).}

\item{delta.design}{[RE models only] Design matrix for delta, defaults to the
\eqn{N \times N}{N x N} identity matrix.}

\item{opt.max}{Should the optimal decision be the maximal treatment effect
(\code{TRUE}, default) or the minimum (\code{FALSE}).}

\item{trt.rank}{Rank of the treatment to derive thresholds for. Defaults to
1, thresholds for the optimum treatment.}

\item{trt.code}{Treatment codings of the reference treatment and in the
parameter vector \eqn{d_k}. Use if treatments re-labelled or re-ordered.
Default is equivalent to \code{1:K}.}

\item{trt.sub}{Only look at thresholds in this subset of treatments in
\code{trt.code}, e.g. if some are excluded from the ranking. Default is
equivalent to \code{1:K}.}

\item{mcid}{Minimal clinically important difference for the decision (when
\code{mcid.type = 'decision'}) or for changing the decision (when
\code{mcid.type = 'change'}). Defaults to 0, use the maximal efficacy
decision rule.}

\item{mcid.type}{Default \code{'decision'}, the decision rule is based on
MCID (see details). Otherwise \code{'change'}, use the maximum efficacy
rule, but only consider changing the decision when the alternative
treatment becomes more effective than the base case by \code{mcid} or more.}
}
\value{
An object of class \code{thresh}.
}
\description{
This function calculates decision-invariant bias-adjustment thresholds and
intervals for Bayesian network meta-analysis, as described by Phillippo
\emph{et al.} (2018). Thresholds are derived from the joint
posterior, and reflect the amount of change to a data point before the
treatment decision changes. Calculation is achieved using fast matrix
operations.
}
\details{
This function provides bias-adjustment threshold analysis for both
  fixed and random effects NMA models, as described by Phillippo \emph{et
  al.} (2018). Parameters \code{mean.dk}, \code{lhood}, and
  \code{post} are always required, however there are differences in the
  specification of \code{post} and other required parameters and between the
  fixed and random effects cases:

  \describe{ \item{\strong{FE models}}{The design matrix \code{X} for basic
  treatment parameters is required. The posterior covariance matrix specified
  in \code{post} should only refer to the basic treatment parameters.}
  \item{\strong{RE models}}{The design matrix \code{mu.design} for additional
  parameters (e.g. covariates) is required, as is the design matrix
  \code{delta.design} for random effects terms. The posterior covariance
  matrix specified in \code{post} should refer to the basic treatment
  parameters, RE terms, and additional parameters \emph{in that order}; i.e.
  \code{post} should be the posterior covariance matrix of the vector
  \eqn{(d^T, \delta^T, \mu^T)^T}.} }
}
\section{Model details}{
 \strong{The FE NMA model}

  The fixed effects NMA model is assumed to be of the form \describe{
  \item{Prior:}{\eqn{d \sim \mathrm{N} ( d_0, \Sigma_d )}{d ~ N(d_0,
  \Sigma_d)}}
  \item{Likelihood:}{\eqn{y|d \sim \mathrm{N} ( \delta, V )}{y|d ~ N(\delta,
  V)}} \item{FE model:}{\eqn{\delta = Xd + M\mu}} }

  The additional parameters \eqn{\mu} may be given any sensible prior; they
  do not affect the threshold analysis in any way.

  \strong{The RE NMA model}

  The random effects NMA model is assumed to be of the form \describe{
  \item{Priors:}{\eqn{ d \sim \mathrm{N} ( d_0, \Sigma_d ), \quad \mu \sim
  \mathrm{N} ( \mu_0, \Sigma_\mu )}{d ~ N(d_0, \Sigma_d), \mu ~ N(\mu_0,
  \Sigma_\mu)}}
  \item{Likelihood:}{\eqn{y|d,\mu,\tau^2 \sim \mathrm{N} ( L\delta + M\mu, V
  )}{y|d,\mu,\tau^2 ~ N(L\delta + M\mu, V)}}
  \item{RE model:}{\eqn{\delta \sim \mathrm{N} ( Xd, \tau^2 )}{\delta ~ N(Xd,
  \tau^2)}} }

  The between-study heterogeneity parameter \eqn{\tau^2} may be given any
  sensible prior. In the RE case, the threshold derivations make the
  approximation that \eqn{\tau^2} is fixed and known.
}

\section{Decision rules}{


  The default decision rule is maximal efficacy; the optimal treatment is
  \eqn{ k^* = \mathrm{argmax}_k \mathrm{E}(d_{k})}{k* = argmax(E(d_k))}.

  When \eqn{\epsilon} = \code{mcid} is greater than zero and
  \code{mcid.type = 'decision'}, the decision rule is no longer for a single
  best treatment, but is based on minimal clinically important difference. A
  treatment is in the optimal set if \eqn{\mathrm{E}(d_k) \ge
  \epsilon}{E(d_k) \ge \epsilon} and \eqn{\max_a \mathrm{E}(d_a) -
  \mathrm{E}(d_k) \le \epsilon}{max E(d_a) - E(d_k) \le \epsilon}.

  When \code{mcid.type = 'change'}, the maximal efficacy rule is used, but
  thresholds are found for when a new treatment is better than the base-case
  optimal by at least \code{mcid}.
}

\examples{
# Please see the vignette "Examples" for worked examples including use of
# this function, including more information on the brief code below.

vignette("Examples", package = "nmathresh")

### Contrast level thresholds for Thrombolytic treatments NMA
K <- 6   # Number of treatments

# Contrast design matrix is
X <- matrix(ncol = K-1, byrow = TRUE,
            c(1, 0, 0, 0, 0,
              0, 1, 0, 0, 0,
              0, 0, 1, 0, 0,
              0, 0, 0, 1, 0,
              0, -1, 1, 0, 0,
              0, -1, 0, 1, 0,
              0, -1, 0, 0, 1))

# Reconstruct hypothetical likelihood covariance matrix using NNLS
lik.cov <- recon_vcov(Thrombo.post.cov, prior.prec = .0001, X = X)

# Thresholds are then
thresh <- nma_thresh(mean.dk = Thrombo.post.summary$statistics[1:(K-1), "Mean"],
                     lhood = lik.cov,
                     post = Thrombo.post.cov,
                     nmatype = "fixed",
                     X = X,
                     opt.max = FALSE)

}
\seealso{
\code{\link{recon_vcov}}, \code{\link{thresh_forest}},
  \code{\link{thresh-class}}.
}
