% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dOcc.R
\name{dOcc}
\alias{dOcc}
\alias{dOcc_s}
\alias{dOcc_v}
\alias{rOcc_s}
\alias{rOcc_v}
\title{Occupancy distribution suitable for use in code{nimble} models}
\usage{
dOcc_s(x, probOcc, probDetect, len = 0, log = 0)

dOcc_v(x, probOcc, probDetect, len = 0, log = 0)

rOcc_s(n, probOcc, probDetect, len = 0)

rOcc_v(n, probOcc, probDetect, len = 0)
}
\arguments{
\item{x}{detection/non-detection vector of 0s (not detected) and 1s
(detected).}

\item{probOcc}{occupancy probability (scalar).}

\item{probDetect}{detection probability (scalar for \code{dOcc_s},
vector for \code{dOcc_v}).}

\item{len}{length of detection/non-detection vector (see below).}

\item{log}{TRUE or 1 to return log probability. FALSE or 0 to
return probability.}

\item{n}{number of random draws, each returning a vector of length
\code{len}. Currently only \code{n = 1} is supported, but the
argument exists for standardization of "\code{r}" functions.}
}
\value{
For \code{dOcc_*}: the probability (or likelihood) or log probability of observation vector \code{x}.

For \code{rOcc_*}: a simulated detection history, \code{x}.
}
\description{
\code{dOcc_*} and \code{rOcc_*} provide occupancy model
distributions that can be used directly from R or in \code{nimble}
models.
}
\details{
These nimbleFunctions provide distributions that can be used directly in R or
in \code{nimble} hierarchical models (via \code{\link[nimble]{nimbleCode}}
and \code{\link[nimble]{nimbleModel}}).

The probability (or likelihood) of observation vector \code{x} depends on
occupancy probability, \code{probOcc}, and detection probability,
\code{probDetect} or \code{probDetect[t]}.

The letter following the 'dOcc_' indicates whether detection probability is
scalar (s, meaning \code{probDetect} is detection probability for every
\code{x[t]}) or vector (v, meaning \code{probDetect[t]} is detection
probabilityfor \code{x[t]}).

When used directly from R, the \code{len} argument to \code{dOcc_*} is not
necessary. It will default to the length of \code{x}.  When used in
\code{nimble} model code (via \code{nimbleCode}), \code{len} must be provided
(even though it may seem redundant).

For more explanation, see
\href{../doc/Introduction_to_nimbleEcology.html}{package vignette} (or
\code{vignette("Introduction_to_nimbleEcology")}).

Compared to writing \code{nimble} models with a discrete latent state for
true occupancy status and a separate scalar datum for each observation, use
of these distributions allows one to directly sum (marginalize) over the
discrete latent state and calculate the probability of all observations from
one site jointly.

These are \code{nimbleFunction}s written in the format of user-defined
distributions for NIMBLE's extension of the BUGS model language. More
information can be found in the NIMBLE User Manual at
\href{https://r-nimble.org}{https://r-nimble.org}.

When using these distributions in a \code{nimble} model, the left-hand side
will be used as \code{x}, and the user should not provide the \code{log}
argument.

For example, in \code{nimble} model code,

\code{detections[i, 1:T] ~ dOcc_s(occupancyProbability,
detectionProbability, T)}

declares that \code{detections[i, 1:T]} (detection history at site \code{i},
for example) follows an occupancy distribution with parameters as indicated,
assuming all the parameters have been declared elsewhere in the model.  This
will invoke (something like) the following call to \code{dOcc_s} when
\code{nimble} uses the model such as for MCMC:

\code{dOcc_s(detections[i, 1:T], occupancyProbability,
detectionProbability, len = T, log = TRUE)}

If an algorithm using a \code{nimble} model with this declaration
needs to generate a random draw for \code{detections[i, 1:T]}, it
will make a similar invocation of \code{rOcc_s}, with \code{n = 1}.

If the detection probabilities are time-dependent, use:

\code{detections[i, 1:T] ~ dOcc_v(occupancyProbability,
detectionProbability[1:T], len = T)}
}
\examples{
\donttest{
# Set up constants and initial values for defining the model
dat <- c(1,1,0,0) # A vector of observations
probOcc <- 0.6
probDetect <- 0.4


# Define code for a nimbleModel
nc <- nimbleCode({
  x[1:4] ~ dOcc_s(probOcc, probDetect, len = 4)
  probOcc ~ dunif(0,1)
  probDetect ~ dunif(0,1)
})

# Build the model, providing data and initial values
Occ_model <- nimbleModel(nc, data = list(x = dat),
                         inits = list(probOcc = probOcc,
                                      probDetect = probDetect))

# Calculate log probability of data from the model
Occ_model$calculate()
# Use the model for a variety of other purposes...
}
}
\seealso{
For dynamic occupancy models, see documentation for
  \code{\link{dDynOcc}}.
}
\author{
Ben Goldstein, Perry de Valpine, and Lauren Ponisio
}
