#' Compute 95% confidence intervals for derived estimates from a matrix
#' population model
#'
#' This function computes the 95% confidence interval for measures derived from
#' a matrix population model using parametric bootstrapping. In this approach a
#' sampling distribution of the matrix population model (MPM) is generated by
#' taking a large number of random independent draws using the sampling
#' distribution of each underlying transition rate. The approach rests on our
#' assumption that survival-related processes are binomial, while fecundity
#' is a Poisson process (see the function `add_mpm_error()` for details).
#'
#' The inputs are the U matrix, which describes the survival-related processes,
#' and the F matrix which describes fecundity. The underlying assumption is
#' that the U matrix is the average of a binomial process while the F matrix is
#' the average of a Poisson process . The confidence interval will depend
#' largely on the sample size used.
#'
#' @param mat_U A matrix that describes the growth and survival process.
#' @param mat_F A matrix that describes fecundity.
#' @param sample_size either (1) a single matrix of sample sizes for each
#'   element of the MPM, (2) a list of two named matrices ("`mat_F_ss`",
#'   "`mat_U_ss`") containing sample sizes for the survival and fecundity
#'   submatrices of the MPM or (3) a single value applied to the whole matrix
#' @param FUN A function to apply to each simulated matrix population model.
#'   This function must take, as input, a single matrix population model (i.e.,
#'   the A matrix). For functions that require only the U matrix, use
#'   `compute_ci_U`.
#' @param ... Additional arguments to be passed to `FUN`.
#' @param n_sim An integer indicating the number of simulations to run. Default
#'   is 1000.
#' @param dist.out Logical. If TRUE, returns a list with both the quantiles and
#'   the simulated estimates. Default is FALSE.
#'
#' @return If dist.out is FALSE, a numeric vector of the 2.5th and 97.5th
#'   quantiles of the estimated measures. If `dist.out = TRUE`, a list with two
#'   elements: `quantiles` and `estimates`. `quantiles` is a numeric vector of
#'   the 2.5th and 97.5th quantiles of the estimated measures, and `estimates`
#'   is a numeric vector of the estimated measures.
#' @references Chapter 12 in Caswell, H. (2001). Matrix Population Models.
#'   Sinauer Associates Incorporated.
#' @family errors
#'
#' @examples
#' set.seed(42) # set seed for repeatability
#'
#' # Data for use in example
#' matU <- matrix(c(
#'   0.1, 0.0,
#'   0.2, 0.4
#' ), byrow = TRUE, nrow = 2)
#'
#' matF <- matrix(c(
#'   0.0, 5.0,
#'   0.0, 0.0
#' ), byrow = TRUE, nrow = 2)
#'
#' set.seed(42)
#'
#' # Example of use to calculate 95% CI of lambda
#' compute_ci(
#'   mat_U = matU, mat_F = matF, sample_size = 10, FUN =
#'     popbio::lambda
#' )
#'
#' # Example of use to calculate 95% CI of generation time
#' compute_ci(
#'   mat_U = matU, mat_F = matF, sample_size = 40, FUN =
#'     popbio::generation.time
#' )
#'
#' # Example of use to calculate 95% CI of generation time and show the
#' # distribution of those bootstrapped estimates
#' xx <- compute_ci(
#'   mat_U = matU, mat_F = matF, sample_size = 100, FUN =
#'     popbio::generation.time, dist.out = TRUE
#' )
#' summary(xx$quantiles)
#' hist(xx$estimates)
#'
#' @importFrom stats quantile
#' @importFrom popdemo eigs
#' @importFrom popbio generation.time
#'
#' @author Owen Jones <jones@biology.sdu.dk>
#'
#' @export
#'
compute_ci <- function(mat_U, mat_F, sample_size, FUN, ...,
                       n_sim = 1000, dist.out = FALSE) {
  # Validation of inputs
  # Check input matrices
  if (!is.matrix(mat_U) || !is.matrix(mat_F)) {
    stop("mat_U and mat_F must be matrices.")
  }
  if (!identical(dim(mat_U), dim(mat_F))) {
    stop("mat_U and mat_F must have the same dimensions.")
  }

  if (!dim(mat_F)[1] == dim(mat_F)[2]) {
    stop("mat_F must be a square matrix.")
  }
  if (!dim(mat_U)[1] == dim(mat_U)[2]) {
    stop("mat_U must be square matrix.")
  }

  # Sample size validation
  if (!(inherits(sample_size, "list") || inherits(sample_size, "matrix") ||
    length(sample_size) == 1)) {
    stop("sample_size needs to be a matrix, a list of two matrices,
         or an integer with length 1")
  }

  # When sample_size is a single matrix.
  if (inherits(sample_size, "matrix")) {
    if (nrow(sample_size) != nrow(mat_U)) {
      stop("if sample_size is a matrix,
           it should be the same dimension as mat_U")
    }
  }

  # When sample_size is a list of two matrices.
  if (inherits(sample_size, "list")) {
    if (!identical(
      lapply(sample_size, dim)[[1]],
      lapply(sample_size, dim)[[2]]
    )) {
      stop("if sample_size is a list of matrices,
           they should both be the same dimensions.")
    }
    if (!identical(lapply(sample_size, dim)[[1]], dim(mat_U))) {
      stop("if sample_size is a list of matrices,
           they should be the same dimension as mat_U")
    }
    if (!sum(names(sample_size) %in% c("mat_F_ss", "mat_U_ss")) == 2) {
      stop("if sample_size is a list of matrices,
           the names of the list entries need to be named
           'mat_F_ss' and 'mat_U_ss'")
    }
  }

  unlisted_sample_size <- unlist(sample_size)

  if (!min(abs(c(unlisted_sample_size %% 1, unlisted_sample_size %% 1 - 1))) <
    .Machine$double.eps^0.5) {
    stop("sample_size must be integer value(s)")
  }

  if (min(unlisted_sample_size) < 0) {
    stop("sample_size must be >= 0.")
  }

  # Check FUN argument
  if (!is.function(FUN)) {
    stop("FUN must be a function.")
  }

  # replicate the simulation of MPMs
  sim_out <- replicate(n_sim, add_mpm_error(mat_U,
    mat_F,
    sample_size,
    split = FALSE,
    by_type = FALSE
  ),
  simplify = FALSE
  )

  # apply the function FUN to each matrix
  estimates <- sapply(sim_out, FUN, ...)

  # Check the estimates and use warnings if necessary
  if (any(is.infinite(estimates))) {
    warning("Some estimates are Inf. \n
            Try running with argument `dist.out = TRUE`
            and examine the estimates.")
  }

  emp_quantiles <- quantile(estimates, c(0.025, 0.975), na.rm = TRUE)

  if (dist.out == FALSE) {
    return(emp_quantiles)
  }
  if (dist.out == TRUE) {
    out <- list(
      "quantiles" = emp_quantiles, "estimates" = estimates,
      "matrices" = sim_out
    )
    return(out)
  }
}
