% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parsnip-window_reg.R
\name{window_reg}
\alias{window_reg}
\title{General Interface for Window Forecast Models}
\usage{
window_reg(mode = "regression", id = NULL, window_size = NULL)
}
\arguments{
\item{mode}{A single character string for the type of model.
The only possible value for this model is "regression".}

\item{id}{An optional quoted column name (e.g. "id") for
identifying multiple time series (i.e. panel data).}

\item{window_size}{A window to apply the window function. By default,
the window uses the full data set, which is rarely the best choice.}
}
\description{
\code{window_reg()} is a way to generate a \emph{specification} of a window model
before fitting and allows the model to be created using
different backends.
}
\details{
A time series window regression is derived using \code{window_reg()}.
The model can be created using the \code{fit()} function using the
following \emph{engines}:
\itemize{
\item \strong{"window_function" (default)} - Performs a Window Forecast
applying a \code{window_function} (engine parameter)
to a window of size defined by \code{window_size}
}
}
\section{Engine Details}{


\strong{function (default engine)}

The engine uses \code{\link[=window_function_fit_impl]{window_function_fit_impl()}}. A time series window function
applies a \code{window_function} to a window of the data (last N observations).
\itemize{
\item The function can return a scalar (single value) or multiple values
that are repeated for each window
\item Common use cases:
\itemize{
\item \strong{Moving Average Forecasts:} Forecast forward a 20-day average
\item \strong{Weighted Average Forecasts:} Exponentially weighting the most recent observations
\item \strong{Median Forecasts:} Forecasting forward a 20-day median
\item \strong{Repeating Forecasts:} Simulating a Seasonal Naive Forecast by
broadcasting the last 12 observations of a monthly dataset into the future
}
}

The key engine parameter is the \code{window_function}. A function / formula:
\itemize{
\item If a function, e.g. \code{mean}, the function is used with
any additional arguments, \code{...} in \code{set_engine()}.
\item If a formula, e.g. \code{~ mean(., na.rm = TRUE)}, it is converted to a function.
}

This syntax allows you to create very compact anonymous functions.
}

\section{Fit Details}{


\strong{Date and Date-Time Variable}

It's a requirement to have a date or date-time variable as a predictor.
The \code{fit()} interface accepts date and date-time features and handles them internally.
\itemize{
\item \code{fit(y ~ date)}
}

\strong{ID features (Multiple Time Series, Panel Data)}

The \code{id} parameter is populated using the \code{fit()} or \code{fit_xy()} function:

\emph{ID Example:} Suppose you have 3 features:
\enumerate{
\item \code{y} (target)
\item \code{date} (time stamp),
\item \code{series_id} (a unique identifer that identifies each time series in your data).
}

The \code{series_id} can be passed to the \code{window_reg()} using
\code{fit()}:
\itemize{
\item \code{window_reg(id = "series_id")} specifes that the \code{series_id} column should be used
to identify each time series.
\item \code{fit(y ~ date + series_id)} will pass \code{series_id} on to the underlying functions.
}

\strong{Window Function Specification (window_function)}

You can specify a function / formula using \code{purrr} syntax.
\itemize{
\item If a function, e.g. \code{mean}, the function is used with
any additional arguments, \code{...} in \code{set_engine()}.
\item If a formula, e.g. \code{~ mean(., na.rm = TRUE)}, it is converted to a function.
}

This syntax allows you to create very compact anonymous functions.

\strong{Window Size Specification (window_size)}

The period can be non-seasonal (\verb{window_size = 1 or "none"}) or
yearly seasonal (e.g. For monthly time stamps, \code{window_size = 12}, \code{window_size = "12 months"}, or \code{window_size = "yearly"}).
There are 3 ways to specify:
\enumerate{
\item \code{window_size = "all"}: A seasonal period is selected based on the periodicity of the data (e.g. 12 if monthly)
\item \code{window_size = 12}: A numeric frequency. For example, 12 is common for monthly data
\item \code{window_size = "1 year"}: A time-based phrase. For example, "1 year" would convert to 12 for monthly data.
}

\strong{External Regressors (Xregs)}

These models are univariate. No xregs are used in the modeling process.
}

\examples{
library(dplyr)
library(parsnip)
library(rsample)
library(timetk)
library(modeltime)

# Data
m750 <- m4_monthly \%>\% filter(id == "M750")
m750

# Split Data 80/20
splits <- initial_time_split(m750, prop = 0.8)

# ---- WINDOW FUNCTION -----

# Used to make:
# - Mean/Median forecasts
# - Simple repeating forecasts

# Median Forecast ----

# Model Spec
model_spec <- window_reg(
        window_size     = 12
    ) \%>\%
    # Extra parameters passed as: set_engine(...)
    set_engine(
        engine          = "window_function",
        window_function = median,
        na.rm           = TRUE
    )

# Fit Spec
model_fit <- model_spec \%>\%
    fit(log(value) ~ date, data = training(splits))
model_fit

# Predict
# - The 12-month median repeats going forward
predict(model_fit, testing(splits))


# ---- PANEL FORECAST - WINDOW FUNCTION ----

# Weighted Average Forecast
model_spec <- window_reg(
        # Specify the ID column for Panel Data
        id          = "id",
        window_size = 12
    ) \%>\%
    set_engine(
        engine = "window_function",
        # Create a Weighted Average
        window_function = ~ sum(tail(.x, 3) * c(0.1, 0.3, 0.6)),
    )

# Fit Spec
model_fit <- model_spec \%>\%
    fit(log(value) ~ date + id, data = training(splits))
model_fit

# Predict: The weighted average (scalar) repeats going forward
predict(model_fit, testing(splits))

# ---- BROADCASTING PANELS (REPEATING) ----

# Simulating a Seasonal Naive Forecast by
# broadcasted model the last 12 observations into the future
model_spec <- window_reg(
        id          = "id",
        window_size = Inf
    ) \%>\%
    set_engine(
        engine          = "window_function",
        window_function = ~ tail(.x, 12),
    )

# Fit Spec
model_fit <- model_spec \%>\%
    fit(log(value) ~ date + id, data = training(splits))
model_fit

# Predict: The sequence is broadcasted (repeated) during prediction
predict(model_fit, testing(splits))

}
\seealso{
\code{\link[=fit.model_spec]{fit.model_spec()}}, \code{\link[=set_engine]{set_engine()}}
}
