% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelStudio.R
\name{modelStudio}
\alias{modelStudio}
\alias{modelStudio.explainer}
\title{Interactive Studio for Explanatory Model Analysis}
\usage{
modelStudio(explainer, ...)

\method{modelStudio}{explainer}(
  explainer,
  new_observation = NULL,
  new_observation_y = NULL,
  facet_dim = c(2, 2),
  time = 500,
  max_features = 10,
  N = 300,
  B = 10,
  eda = TRUE,
  show_info = TRUE,
  parallel = FALSE,
  options = ms_options(),
  viewer = "external",
  ...
)
}
\arguments{
\item{explainer}{An \code{explainer} created with \code{DALEX::explain()}.}

\item{...}{Other parameters.}

\item{new_observation}{New observations with columns that correspond to variables used in the model.}

\item{new_observation_y}{True label for \code{new_observation} (optional).}

\item{facet_dim}{Dimensions of the grid. Default is \code{c(2,2)}.}

\item{time}{Time in ms. Set the animation length. Default is \code{500}.}

\item{max_features}{Maximum number of features to be included in BD and SV plots.
Default is \code{10}.}

\item{N}{Number of observations used for the calculation of PD and AD.
\code{10*N} is a number of observations used for the calculation of FI.
Default \code{N} is \code{300}.
See \href{https://modelstudio.drwhy.ai/articles/ms-perks-features.html#more-calculations-means-more-time}{\bold{vignette}}}

\item{B}{Number of permutation rounds used for calculation of SV and FI.
Default is \code{10}.
See \href{https://modelstudio.drwhy.ai/articles/ms-perks-features.html#more-calculations-means-more-time}{\bold{vignette}}}

\item{eda}{Compute EDA plots. Default is \code{TRUE}.}

\item{show_info}{Verbose a progress on the console. Default is \code{TRUE}.}

\item{parallel}{Speed up the computation using \code{parallelMap::parallelMap()}.
See \href{https://modeloriented.github.io/modelStudio/articles/ms-perks-features.html#parallel-computation}{\bold{vignette}}.
This might interfere with showing progress using \code{show_info}.}

\item{options}{Customize \code{modelStudio}. See \code{\link{ms_options}} and
\href{https://modeloriented.github.io/modelStudio/articles/ms-perks-features.html#plot-options}{\bold{vignette}}.}

\item{viewer}{Default is \code{external} to display in an external RStudio window.
Use \code{browser} to display in an external browser or
\code{internal} to use the RStudio internal viewer pane for output.}
}
\value{
An object of the \code{r2d3, htmlwidget, modelStudio} class.
}
\description{
This function computes various (instance and dataset level) model explanations and produces an interactive,
customisable dashboard. It consists of multiple panels for plots with their short descriptions.
Easily save and share the HTML dashboard with others. Tools for model exploration unite with tools for EDA
(Exploratory Data Analysis) to give a broad overview of the model behavior.

Theoretical introduction to the plots:
\href{https://pbiecek.github.io/ema/}{Explanatory Model Analysis: Explore, Explain and Examine Predictive Models}
}
\examples{
library("DALEX")
library("modelStudio")

#:# ex1 classification on 'titanic' data

# fit a model
model_titanic <- glm(survived ~., data = titanic_imputed, family = "binomial")

# create an explainer for the model
explainer_titanic <- explain(model_titanic,
                             data = titanic_imputed,
                             y = titanic_imputed$survived,
                             label = "Titanic GLM")

# pick observations
new_observations <- titanic_imputed[1:2,]
rownames(new_observations) <- c("Lucas","James")

# make a studio for the model
modelStudio(explainer_titanic,
            new_observations)

\donttest{

#:# ex2 regression on 'apartments' data
library("ranger")

model_apartments <- ranger(m2.price ~. ,data = apartments)

explainer_apartments <- explain(model_apartments,
                                data = apartments,
                                y = apartments$m2.price)

new_apartments <- apartments[1:2,]
rownames(new_apartments) <- c("ap1","ap2")

# change dashboard dimensions and animation length
modelStudio(explainer_apartments,
            new_apartments,
            facet_dim = c(2, 3),
            time = 800)

# add information about true labels
modelStudio(explainer_apartments,
            new_apartments,
            new_observation_y = new_apartments$m2.price)

# don't compute EDA plots
modelStudio(explainer_apartments,
            eda = FALSE)


#:# ex3 xgboost model on 'HR' dataset
library("xgboost")

HR_matrix <- model.matrix(status == "fired" ~ . -1, HR)

# fit a model
xgb_matrix <- xgb.DMatrix(HR_matrix, label = HR$status == "fired")
params <- list(max_depth = 3, objective = "binary:logistic", eval_metric = "auc")
model_HR <- xgb.train(params, xgb_matrix, nrounds = 300)

# create an explainer for the model
explainer_HR <- explain(model_HR,
                        data = HR_matrix,
                        y = HR$status == "fired",
                        label = "xgboost")

# pick observations
new_observation <- HR_matrix[1:2, , drop=FALSE]
rownames(new_observation) <- c("id1", "id2")
# make a studio for the model
modelStudio(explainer_HR,
            new_observation)

}

}
\references{
\itemize{
  \item The input object is implemented in \href{https://modeloriented.github.io/DALEX/}{\bold{DALEX}}
  \item Feature Importance, Ceteris Paribus, Partial Dependence and Accumulated Dependence plots
   are implemented in \href{https://modeloriented.github.io/ingredients/}{\bold{ingredients}}
  \item Break Down and Shapley Values plots are implemented in
   \href{https://modeloriented.github.io/iBreakDown/}{\bold{iBreakDown}}
}
}
\seealso{
Vignettes: \href{https://modeloriented.github.io/modelStudio/articles/ms-r-python-examples.html}{\bold{modelStudio - R & Python examples}}
and \href{https://modeloriented.github.io/modelStudio/articles/ms-perks-features.html}{\bold{modelStudio - perks and features}}
}
