% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sb.r
\name{scoreBased}
\alias{scoreBased}
\title{Score based variance estimation for multiple imputation}
\usage{
scoreBased(imps, analysisFun, scoreFun, pd = NULL, dfComplete = NULL,
  ...)
}
\arguments{
\item{imps}{A list of imputed datasets produced by one of the imputation functions
in \code{mlmi} or another package.}

\item{analysisFun}{A function to analyse the imputed datasets that when applied to
a dataset returns a list containing a vector \code{est}.}

\item{scoreFun}{A function whose first argument is a dataset and whose second argument is
a vector of parameter values. It should return a matrix of subject level scores
evaluated at the parameter value passed to it.}

\item{pd}{If \code{imps} was not generated by one of the imputation functions in
\code{mlmi}, this argument must be specified to indicate whether the imputations
were generated using posterior draws (TRUE) or not (FALSE).}

\item{dfComplete}{The complete data degrees of freedom. If \code{analysisFun} returns a vector
of parameter estimates, \code{dfComplete} should be a vector of the same length. If not
specified, it is assumed that the complete data degrees of freedom is effectively infinite (1e+05).}

\item{...}{Other parameters that are to be passed through to \code{analysisFun}.}
}
\value{
A list containing the overall parameter estimates, its corresponding covariance matrix, and
degrees of freedom for each parameter.
}
\description{
This function implements the score based variance estimation approach described by von Hippel
(2018), which is based on earlier work by Wang and Robins (1998).
}
\examples{
#simulate a partially observed dataset
set.seed(1234)
n <- 100
x <- rnorm(n)
y <- x+rnorm(n)
y[1:50] <- NA
temp <- data.frame(x,y)
#impute using normUniImp, without posterior draws
imps <- normUniImp(temp, y~x, M=10, pd=FALSE)

#define a function which performs our desired analysis on a dataset, returning
#the parameter estimates
yonx <- function(inputData) {
  fitmod <- lm(y~x, data=inputData)
  list(est=c(fitmod$coef,sigma(fitmod)^2))
}

#define a function which when passed a dataset and parameter
#vector, calculates the likelihood score vector
myScore <- function(inputData, parm) {
  beta0 <- parm[1]
  beta1 <- parm[2]
  sigmasq <- parm[3]
  res <- inputData$y - beta0 - beta1*inputData$x
  cbind(res/sigmasq, (res*inputData$x)/sigmasq, res^2/(2*sigmasq^2)-1/(2*sigmasq))
}

#call scoreBased to perform variance estimation
scoreBased(imps, analysisFun=yonx, scoreFun=myScore)
}
\references{
Wang N., Robins J.M. (1998) Large-sample theory for parametric multiple imputation procedures.
Biometrika 85(4): 935-948. \href{https://doi.org/10.1093/biomet/85.4.935}{https://doi.org/10.1093/biomet/85.4.935}.

von Hippel P.T. (2018) Maximum likelihood multiple imputation: faster,
more efficient imputation without posterior draws. \href{https://arxiv.org/abs/1210.0870v9}{arXiv:1210.0870v9}.
}
