% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fixedCalib.R
\name{fixedCalib}
\alias{fixedCalib}
\title{Fixed-item calibration method}
\usage{
fixedCalib(
  data,
  model = 1,
  old_mod,
  PAU = "MWU",
  NEMC = "MEM",
  technical = list(),
  ...
)
}
\arguments{
\item{data}{new data to be used for calibration. Note that to be consistent
with the \code{mod} object, observed responses/NA placeholders must be included
to link the item names used in the original \code{mod} definition
(i.e., \code{extract.mirt(mod, what = 'itemnames')})}

\item{model}{type of model to fit for the complete dataset (not that for the fixed items
in \code{old_mod} the factor loadings/constraints specified by the potential \code{\link{mirt.model}}
specification is not relevant)}

\item{old_mod}{a model of class SingleGroupClass fitted using \code{\link{mirt}}}

\item{PAU}{prior ability update (PAU) approach. Supports none (\code{"NWU"}),
one (\code{"OWU"}), and many (\code{"MWU"})}

\item{NEMC}{number of EM cycles (NEMC) to use for the to-be-estimated parameters.
Supports one (\code{"OEM"}) and many (\code{"MEM"})}

\item{technical}{list of technical estimation arguments
(see \code{\link{mirt}} for details)}

\item{...}{additional arguments to pass to \code{\link{mirt}}}
}
\description{
Implements the set of fixed-item calibration methods described by Kim (2006). The initial
calibrated model must be fitted via \code{\link{mirt}}, is currently limited to
unidimensional models only, and should only be utilized when the new set of responses
are obtained from a population with similar distributional characteristics in the latent traits.
For more flexible calibration of items, including a fixed-item calibration variant involving
anchor items for equating, see \code{\link{multipleGroup}}.
}
\examples{
\dontrun{

# single factor
set.seed(12345)
J <- 50
a <- matrix(abs(rnorm(J,1,.3)), ncol=1)
d <- matrix(rnorm(J,0,.7),ncol=1)
itemtype <- rep('2PL', nrow(a))

# calibration data theta ~ N(0,1)
N <- 3000
dataset1 <- simdata(a, d, N = N, itemtype=itemtype)

# new data (again, theta ~ N(0,1))
dataset2 <- simdata(a, d, N = 1000, itemtype=itemtype)

# last 40\% of experimental items not given to calibration group
#     (unobserved; hence removed)
dataset1 <- dataset1[,-c(J:(J*.6))]
head(dataset1)

# assume first 60\% of items not given to new group
dataset2[,colnames(dataset1)] <- NA
head(dataset2)

#--------------------------------------

# calibrated model from dataset1 only
mod <- mirt(dataset1, model = 1)
coef(mod, simplify=TRUE)

# No Prior Weights Updating and One EM Cycle (NWU-OEM)
NWU_OEM <- fixedCalib(dataset2, model = 1, old_mod = mod, PAU = 'NWU', NEMC = 'OEM')
coef(NWU_OEM, simplify=TRUE)
data.frame(coef(NWU_OEM, simplify=TRUE)$items[,c('a1','d')], pop_a1=a, pop_d=d)
plot(NWU_OEM, type = 'empiricalhist')

# No Prior Weights Updating and Multiple EM Cycles (NWU-MEM)
NWU_MEM <- fixedCalib(dataset2, model = 1, old_mod = mod, PAU = 'NWU')
coef(NWU_MEM, simplify=TRUE)
data.frame(coef(NWU_MEM, simplify=TRUE)$items[,c('a1','d')], pop_a1=a, pop_d=d)
plot(NWU_MEM, type = 'empiricalhist')

# One Prior Weights Updating and One EM Cycle (OWU-OEM)
OWU_OEM <- fixedCalib(dataset2, model = 1, old_mod = mod, PAU = 'OWU', NEMC = "OEM")
coef(OWU_OEM, simplify=TRUE)
data.frame(coef(OWU_OEM, simplify=TRUE)$items[,c('a1','d')], pop_a1=a, pop_d=d)
plot(OWU_OEM, type = 'empiricalhist')

# One Prior Weights Updating and Multiple EM Cycles (OWU-MEM)
OWU_MEM <- fixedCalib(dataset2, model = 1, old_mod = mod, PAU = 'OWU')
coef(OWU_MEM, simplify=TRUE)
data.frame(coef(OWU_MEM, simplify=TRUE)$items[,c('a1','d')], pop_a1=a, pop_d=d)
plot(OWU_MEM, type = 'empiricalhist')

# Multiple Prior Weights Updating and Multiple EM Cycles (MWU-MEM)
MWU_MEM <- fixedCalib(dataset2, model = 1, old_mod = mod)
coef(MWU_MEM, simplify=TRUE)
data.frame(coef(MWU_MEM, simplify=TRUE)$items[,c('a1','d')], pop_a1=a, pop_d=d)
plot(MWU_MEM, type = 'empiricalhist')

}
}
\references{
Kim, S. (2006). A comparative study of IRT fixed parameter calibration methods.
\emph{Journal of Educational Measurement, 4}(43), 355-381.
}
\seealso{
\code{\link{mirt}}, \code{\link{multipleGroup}}
}
