% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mig_chord.R
\name{mig_chord}
\alias{mig_chord}
\title{Chord diagram for directional origin-destination data}
\usage{
mig_chord(
  x,
  lab = NULL,
  lab_bend1 = NULL,
  lab_bend2 = NULL,
  label_size = 1,
  label_nudge = 0,
  axis_size = 0.8,
  axis_breaks = NULL,
  ...,
  no_labels = FALSE,
  no_axis = FALSE,
  clear_circos_par = TRUE,
  zero_margin = TRUE,
  start.degree = 90,
  gap.degree = 4,
  track.margin = c(-0.1, 0.1),
  points.overflow.warning = FALSE
)
}
\arguments{
\item{x}{Data frame with origin in first column, destination in second column and bilateral measure in third column}

\item{lab}{Named vector of labels for plot. If \code{NULL} will use names from \code{d}}

\item{lab_bend1}{Named vector of bending labels for plot. Note line breaks do not work with \code{facing = "bending"} in circlize.}

\item{lab_bend2}{Named vector of second row of bending labels for plot.}

\item{label_size}{Font size of label text.}

\item{label_nudge}{Numeric value to nudge labels towards (negative number) or away (positive number) the sector axis.}

\item{axis_size}{Font size on axis labels.}

\item{axis_breaks}{Numeric value for how often to add axis label breaks. Default not activated, uses default from \code{circlize::circos.axis()}}

\item{...}{Arguments for \code{circlize::chordDiagramFromDataFrame()}.}

\item{no_labels}{Logical to indicate if to include plot labels. Set to \code{FALSE} by default.}

\item{no_axis}{Logical to indicate if to include plot axis. Set to \code{FALSE} by default.}

\item{clear_circos_par}{Logical to run \code{circlize::circos.clear()}. Set to \code{TRUE} by default. Set to \code{FALSE} if you wish to add further to the plot.}

\item{zero_margin}{Set margins of the plotting graphics device to zero. Set to \code{TRUE} by default.}

\item{start.degree}{Argument for \code{circlize::circos.par()}.}

\item{gap.degree}{Argument for \code{circlize::chordDiagramFromDataFrame()}.}

\item{track.margin}{Argument for \code{circlize::chordDiagramFromDataFrame()}.}

\item{points.overflow.warning}{Argument for \code{circlize::chordDiagramFromDataFrame()}.}
}
\value{
Chord diagram based on first three columns of \code{x}. The function tweaks the defaults of \code{circlize::chordDiagramFromDataFrame()} for easier plotting of directional origin-destination data. Users can override these defaults and pass additional tweaks using any of the \code{circlize::chordDiagramFromDataFrame()} arguments.

The layout of the plots are designed to specifically work on plotting images into PDF devices with widths and heights of 7 inches (the default dimension when using the \code{pdf} function). See the end of the examples for converting PDFs to images. 

Fitting all the labels on the page is usually the most time consuming task. Use the different label options, including line breaks, \code{label_nudge}, track height in \code{preAllocateTracks} and font sizes in \code{label_size} and \code{axis_size} to find the best fit. If none of the label options produce desirable results, plot your own using \code{circlize::circos.text} having set \code{no_labels = TRUE} and \code{clear_circos_par = FALSE}.
}
\description{
Adaption of \code{circlize::chordDiagramFromDataFrame()} with defaults set to allow for more effective visualisation of directional origin-destination data
}
\examples{
\dontrun{
library(tidyverse)
library(countrycode)
# download Abel and Cohen (2019) estimates
f <- read_csv("https://ndownloader.figshare.com/files/26239945")

# use dictionary to get region to region flows
d <- f \%>\%
  mutate(
    orig = countrycode(
      sourcevar = orig, custom_dict = dict_ims,
      origin = "iso3c", destination = "region"),
    dest = countrycode(
      sourcevar = dest, custom_dict = dict_ims,
      origin = "iso3c", destination = "region")
  ) \%>\%
group_by(year0, orig, dest) \%>\%
summarise_all(sum) \%>\%
ungroup()
d

# 2015-2020 pseudo-Bayesian estimates for plotting
pb <- d \%>\%
    filter(year0 == 2015) \%>\%
    mutate(flow = da_pb_closed/1e6) \%>\%
    select(orig, dest, flow)
    
# pdf(file = "chord.pdf")
mig_chord(x = pb)
# dev.off()
# file.show("chord.pdf")

# pass arguments to circlize::chordDiagramFromDataFrame
# pdf(file = "chord.pdf")
mig_chord(x = pb, 
          # order of regions
          order = rev(unique(pb$orig)),
          # spacing for labels
          preAllocateTracks = list(track.height = 0.3),
          # colours
          grid.col = c("blue", "royalblue", "navyblue", "skyblue", "cadetblue", "darkblue")
          ) 
# dev.off()
# file.show("chord.pdf")

# multiple line labels to fit on longer labels
r <- pb \%>\%
  sum_region() \%>\%
  mutate(lab = str_wrap_n(string = region, n = 2)) \%>\%
  separate(col = lab, into = c("lab1", "lab2"), sep = "\n", remove = FALSE, fill = "right")
r

# pdf(file = "chord.pdf")
mig_chord(x = pb,
          lab = r \%>\%
            select(region, lab) \%>\%
            deframe(),
          preAllocateTracks = list(track.height = 0.25),
          label_size = 0.8,
          axis_size = 0.7
          )
# dev.off()
# file.show("chord.pdf")

# bending labels
# pdf(file = "chord.pdf")
mig_chord(x = pb,
          lab_bend1 = r \%>\%
            select(region, lab1) \%>\%
            deframe(),
          lab_bend2 = r \%>\%
            select(region, lab2) \%>\%
            deframe()
          )
# dev.off()
# file.show("chord.pdf")


# convert pdf to image file
# library(magick)
# p <- image_read_pdf("chord.pdf")
# image_write(image = p, path = "chord.png")
# file.show("chord.png")
}
}
