% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mies_methods.R
\name{mies_generate_offspring}
\alias{mies_generate_offspring}
\title{Generate Offspring Through Mutation and Recombination}
\usage{
mies_generate_offspring(
  inst,
  lambda,
  parent_selector = NULL,
  mutator = NULL,
  recombinator = NULL,
  budget_id = NULL
)
}
\arguments{
\item{inst}{(\code{\link[bbotk:OptimInstance]{OptimInstance}})\cr
Optimization instance to evaluate.}

\item{lambda}{(\code{integer(1)})\cr
Number of new individuals to generate. This is not necessarily the number with which \code{parent_selector}
gets called, because \code{recombinator} could in principle need more than \code{lambda} input individuals to
generate \code{lambda} output individuals.}

\item{parent_selector}{(\code{\link{Selector}} | \code{NULL})\cr
\code{\link{Selector}} operator that selects parent individuals depending on configuration values
and objective results. When \code{parent_selector$operate()} is called, then objectives that
are being minimized are multiplied with -1 (through \code{\link[=mies_get_fitnesses]{mies_get_fitnesses()}}), since \code{\link{Selector}}s always try to maximize fitness.
When this is \code{NULL} (default), then a \code{\link{SelectorBest}} us used.\cr
The \code{\link{Selector}} must be primed on a superset of \code{inst$search_space}; this \emph{includes} the "budget" component
when performing multi-fidelity optimization. All components on which \code{selector} is primed on must occur in the archive.\cr
The given \code{\link{Selector}} \emph{may} return duplicates.}

\item{mutator}{(\code{\link{Mutator}} | \code{NULL})\cr
\code{\link{Mutator}} operation to apply to individuals selected out of \code{inst} using \code{parent_selector}.\cr
The \code{\link{Mutator}} must be primed on a \code{\link[paradox:ParamSet]{ParamSet}} similar to \code{inst$search_space},
but \emph{without} the "budget" component when \code{budget_id} is given (multi-fidelity optimization). Such a
\code{\link[paradox:ParamSet]{ParamSet}} can be generated for example using \code{\link{mies_prime_operators}}.\cr
When this is \code{NULL} (default), then a \code{\link{MutatorNull}} is used, effectively disabling mutation.}

\item{recombinator}{(\code{\link{Recombinator}} | \code{NULL})\cr
\code{\link{Recombinator}} operation to apply to individuals selected out of \code{int} using \code{parent_selector} after mutation using \code{mutator}.
The \code{\link{Recombinator}} must be primed on a \code{\link[paradox:ParamSet]{ParamSet}} similar to \code{inst$search_space},
but \emph{without} the "budget" component when \code{budget_id} is given (multi-fidelity optimization). Such a
\code{\link[paradox:ParamSet]{ParamSet}} can be generated for example using \code{\link{mies_prime_operators}}.\cr
When this is \code{NULL} (default), then a \code{\link{RecombinatorNull}} is used, effectively disabling recombination.}

\item{budget_id}{(\code{character(1)} | \code{NULL})\cr
Budget compnent when doing multi-fidelity optimization. This component of the search space is removed from
individuals sampled from the archive in \code{inst} before giving it to \code{mutator} and \code{recombinator}.
Should be \code{NULL} when not doing multi-fidelity.}
}
\value{
\code{\link[data.table:data.table]{data.table}}: A table of configurations proposed as offspring to be evaluated
using \code{\link[=mies_evaluate_offspring]{mies_evaluate_offspring()}}.
}
\description{
Generate new proposal individuals to be evaluated using \code{\link[=mies_evaluate_offspring]{mies_evaluate_offspring()}}.

Parent individuals are selected using \code{parent_selector}, then mutated using \code{mutator}, and thend
recombined using \code{recombinator}. If only a subset of these operations is desired, then
it is possible to set \code{mutator} or \code{recombinator} to the respective "null"-operators.
}
\examples{
set.seed(1)

library("bbotk")
lgr::threshold("warn")

# Define the objective to optimize
objective <- ObjectiveRFun$new(
  fun = function(xs) {
    z <- exp(-xs$x^2 - xs$y^2) + 2 * exp(-(2 - xs$x)^2 - (2 - xs$y)^2)
    list(Obj = z)
  },
  domain = ps(x = p_dbl(-2, 4), y = p_dbl(-2, 4)),
  codomain = ps(Obj = p_dbl(tags = "maximize"))
)

# Get a new OptimInstance
oi <- OptimInstanceSingleCrit$new(objective,
  terminator = trm("evals", n_evals = 100)
)

# Demo operators
m = mut("gauss", sdev = 0.1)
r = rec("xounif")
s = sel("random")
# Operators must be primed
mies_prime_operators(objective$domain, list(m), list(r), list(s))

# We would normally call mies_init_population, but for reproducibility
# we are going to evaluate three given points

oi$eval_batch(data.table::data.table(x = 0:2, y = 2:0, dob = 1, eol = NA_real_))

# Evaluated points:
oi$archive

# Use default operators: no mutation, no recombination, parent_selctor is
# sel("best") --> get one individual, the one with highest performance in the
# archive (x = 1, y = 1).
# (Note 'mies_generate_offspring()' does not modify 'oi')
mies_generate_offspring(oi, lambda = 1)

# Mutate the selected individual after selection. 'm' has 'sdev' set to 0.1,
# so the (x = 1, y = 1) is slightly permuted.
mies_generate_offspring(oi, lambda = 1, mutator = m)

# Recombination, then mutation.
# Even though lambda is 1, there will be two individuals selected with
# sel("best") and recombined, because rec("xounif") needs two parents. One
# of the crossover results is discarded (respecting that 'lambda' is 1),
# the other is mutated and returned.
mies_generate_offspring(oi, lambda = 1, mutator = m, recombinator = r)

# General application: select, recombine, then mutate.
mies_generate_offspring(oi, lambda = 5, parent_selector = s, mutator = m, recombinator = r)

}
\seealso{
Other mies building blocks: 
\code{\link{mies_evaluate_offspring}()},
\code{\link{mies_get_fitnesses}()},
\code{\link{mies_init_population}()},
\code{\link{mies_select_from_archive}()},
\code{\link{mies_step_fidelity}()},
\code{\link{mies_survival_comma}()},
\code{\link{mies_survival_plus}()}
}
\concept{mies building blocks}
