% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sjl.R
\name{sjl}
\alias{sjl}
\alias{sjl_rel}
\title{Compute MCTQ social jetlag}
\usage{
sjl(msw, msf, abs = TRUE, method = "shorter")

sjl_rel(msw, msf, method = "shorter")
}
\arguments{
\item{msw}{An \code{\link[hms:hms]{hms}} object corresponding to the \strong{local time of
mid-sleep on workdays} from a standard, micro, or shift version of the
MCTQ questionnaire. You can use \code{\link[=msl]{msl()}} to compute it.}

\item{msf}{An \code{\link[hms:hms]{hms}} object corresponding to the \strong{local time of
mid-sleep on work-free days} from a standard, micro, or shift version of
the MCTQ questionnaire. You can use \code{\link[=msl]{msl()}} to compute it.}

\item{abs}{(optional) a \code{\link[base:logical]{logical}} object indicating if the
function must return an absolute value (default: \code{TRUE}).}

\item{method}{(optional) a string indicating which method the function must
use to compute the social jetlag. See the Methods section to learn
more (default: \code{"shorter"}).}
}
\value{
\itemize{
\item If \code{abs = TRUE}, a \code{\link[lubridate:duration]{Duration}} object corresponding
to the absolute social jetlag.
\item If \code{abs = FALSE}, a \code{\link[lubridate:duration]{Duration}} object
corresponding to the relative social jetlag.
}

The output may also vary depending on the \code{method} used.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{sjl()} computes the \strong{relative or absolute social jetlag} for standard,
micro, and shift versions of the Munich ChronoType Questionnaire (MCTQ).

\code{sjl_rel()} is just a wrapper for \code{sjl()} with \code{abs = FALSE}.
}
\details{
\strong{Standard MCTQ} functions were created following the guidelines in
Roenneberg, Wirz-Justice, & Merrow (2003), Roenneberg, Allebrandt, Merrow, &
Vetter (2012), and from The Worldwide Experimental Platform (theWeP, n.d.).

\strong{\eqn{\mu}MCTQ} functions were created following the guidelines in Ghotbi
et al. (2020), in addition to the guidelines used for the standard MCTQ.

\strong{MCTQ\eqn{^{Shift}}{ Shift}} functions were created following the
guidelines in Juda, Vetter, & Roenneberg (2013), in addition to the
guidelines used for the standard MCTQ.

See the References section to learn more.
\subsection{Class requirements}{

The \code{mctq} package works with a set of object classes specially created to
hold time values. These classes can be found in the
\link[lubridate:lubridate-package]{lubridate} and \link[hms:hms-package]{hms}
packages. Please refer to those package documentations to learn more about
them.
}

\subsection{Rounding and fractional time}{

Some operations may produce an output with fractional time (e.g.,
\code{"19538.3828571429s (~5.43 hours)"}, \code{01:15:44.505}). If you want, you
can round it with \code{\link[=round_time]{round_time()}}.

Our recommendation is to avoid rounding, but, if you do, make sure that you
only round your values after all computations are done. That way you avoid
\href{https://en.wikipedia.org/wiki/Round-off_error}{round-off errors}.
}
}
\section{Guidelines}{


Roenneberg, Allebrandt, Merrow, & Vetter (2012), Juda, Vetter, & Roenneberg
(2013), and The Worldwide Experimental Platform (n.d.) guidelines for \code{sjl()}
(\eqn{SJL_{rel}}{SJL_rel} and \eqn{SJL}) computation are as follows.
\subsection{Notes}{
\itemize{
\item For MCTQ\eqn{^{Shift}}{ Shift}, the computation below must be applied to
each shift section of the questionnaire.
\item Due to time arithmetic issues, \code{sjl()} does a slightly different
computation by default than those proposed by the authors mentioned above.
See \code{vignette("sjl-computation", package = "mctq")} for more details.
\item If you are visualizing this documentation in plain text, you may have some
trouble understanding the equations. You can see this documentation on the
package \href{https://docs.ropensci.org/mctq/reference/}{website}.
}
}

\subsection{For standard and micro versions of the MCTQ}{

Relative social jetlag (\eqn{SJL_{rel}}{SJL_rel}):

\strong{\deqn{MSF - MSW}}

Absolute social jetlag (\eqn{SJL}):

\strong{\deqn{| MSF - MSW |}}

Where:
\itemize{
\item \eqn{MSW} = Local time of mid-sleep on workdays.
\item \eqn{MSF} = Local time of mid-sleep on work-free days.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days.
}

\subsection{For the shift version of the MCTQ}{

Relative social jetlag (\eqn{SJL_{rel}}{SJL_rel}):

\strong{\deqn{MSF^{M/E/N} - MSW^{M/E/N}}{MSF_M/E/N - MSW_M/E/N}}

Absolute social jetlag (\eqn{SJL}):

\strong{\deqn{| MSF^{M/E/N} - MSW^{M/E/N} |}{| MSF_M/E/N - MSW_M/E/N |}}

Where:
\itemize{
\item \eqn{MSW^{M/E/N}}{MSW_M/E/N} = Local time of mid-sleep between two days in
a particular shift.
\item \eqn{MSF^{M/E/N}}{MSF_M/E/N} = Local time of mid-sleep between two free
days after a particular shift.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days, \eqn{M} =
Morning shift; \eqn{E} = Evening shift; \eqn{N} = Night shift.
}
}

\section{Methods for computing the social jetlag}{


There are different approaches to compute the social jetlag (\eqn{SJL}). By
default, \code{sjl()} uses an approach that we call "the shorter interval
approach" (\code{"shorter"}).

The topics below provide a simple explanation of each method supported by
\code{sjl()}. To get a detail understating of this methods, see
\code{vignette("sjl-computation", package = "mctq")}.
\itemize{
\item \code{"difference"}
}

By using \code{method = "difference"}, \code{sjl()} will do the exact computation
proposed by the MCTQ authors, i.e., \eqn{SJL} will be computed as the linear
difference between \eqn{MSF} and \eqn{MSW} (see the Guidelines section).

\strong{We do not recommend using this method}, as it has many limitations.
\itemize{
\item \code{"shorter"}
}

This is the default method for \code{sjl()}. It's based on the shorter
interval between \eqn{MSW} and \eqn{MSF}, solving most of the issues
relating to \eqn{SJL} computation.
\itemize{
\item \code{"longer"}
}

The \code{"longer"} method uses the same logic of the \code{"shorter"} method, but,
instead of using the shorter interval between \eqn{MSW} and \eqn{MSF}, it
uses the longer interval between the two, considering a two-day window.

This method may help in special contexts, like when dealing with
shift-workers that have a greater than 12 hours distance between their
mid-sleep hours.
}

\examples{
## Scalar example

msw <- hms::parse_hm("03:30")
msf <- hms::parse_hm("05:00")

sjl(msw, msf)
#> [1] "5400s (~1.5 hours)" # Expected
sjl(msw, msf, abs = FALSE)
#> [1] "5400s (~1.5 hours)" # Expected
sjl_rel(msw, msf) # Wrapper function
#> [1] "5400s (~1.5 hours)" # Expected

msw <- hms::parse_hm("04:30")
msf <- hms::parse_hm("23:30")

sjl(msw, msf)
#> [1] "18000s (~5 hours)" # Expected
sjl(msw, msf, abs = FALSE)
#> [1] "18000s (~-5 hours)" # Expected
sjl_rel(msw, msf) # Wrapper function
#> [1] "18000s (~-5 hours)" # Expected

msw <- hms::as_hms(NA)
msf <- hms::parse_hm("05:15")

sjl(msw, msf)
#> [1] NA # Expected

## Vector example

msw <- c(hms::parse_hm("02:05"), hms::parse_hm("04:05"))
msf <- c(hms::parse_hm("23:05"), hms::parse_hm("04:05"))

sjl(msw, msf)
#> [1] "10800s (~3 hours)" "0s" # Expected
sjl(msw, msf, abs = FALSE)
#> [1] "-10800s (~-3 hours)" "0s" # Expected
sjl_rel(msw, msf) # Wrapper function
#> [1] "-10800s (~-3 hours)" "0s" # Expected

## Using different methods

msw <- hms::parse_hm("19:15")
msf <- hms::parse_hm("02:30")

sjl(msw, msf, abs = FALSE, method = "difference")
#> [1] "-60300s (~-16.75 hours)" # Expected
sjl(msw, msf, abs = FALSE, method = "shorter") # Default method
#> [1] "26100s (~7.25 hours)" # Expected
sjl(msw, msf, abs = FALSE, method = "longer")
#> [1] "-60300s (~-16.75 hours)" # Expected

msw <- hms::parse_hm("02:45")
msf <- hms::parse_hm("04:15")

sjl(msw, msf, abs = FALSE, method = "difference")
#> [1] "5400s (~1.5 hours)" # Expected
sjl(msw, msf, abs = FALSE, method = "shorter") # Default method
#> [1] "5400s (~1.5 hours)" # Expected
sjl(msw, msf, abs = FALSE, method = "longer")
#> [1] "-81000s (~-22.5 hours)" # Expected

## Converting the output to 'hms'

msw <- hms::parse_hm("01:15")
msf <- hms::parse_hm("03:25")
sjl(msw, msf)
#> [1] "7800s (~2.17 hours)" # Expected

hms::as_hms(as.numeric(sjl(msw, msf)))
#> 02:10:00 # Expected

## Rounding the output at the seconds level

msw <- hms::parse_hms("04:19:33.1234")
msf <- hms::parse_hms("02:55:05")
sjl(msw, msf)
#> [1] "5068.12339997292s (~1.41 hours)" # Expected

round_time(sjl(msw, msf))
#> [1] "5068s (~1.41 hours)" # Expected
}
\references{
Ghotbi, N., Pilz, L. K., Winnebeck, E. C., Vetter, C., Zerbini, G., Lenssen,
D., Frighetto, G., Salamanca, M., Costa, R., Montagnese, S., & Roenneberg, T.
(2020). The \eqn{\mu}MCTQ: an ultra-short version of the Munich ChronoType
Questionnaire. \emph{Journal of Biological Rhythms}, \emph{35}(1), 98-110.
\doi{10.1177/0748730419886986}.

Jankowski K. S. (2017). Social jet lag: sleep-corrected formula.
\emph{Chronobiology International}, \emph{34}(4), 531-535.
\doi{10.1080/07420528.2017.1299162}.

Juda, M., Vetter, C., & Roenneberg, T. (2013). The Munich ChronoType
Questionnaire for shift-workers (MCTQ\eqn{^{Shift}}{ Shift}). \emph{Journal of
Biological Rhythms}, \emph{28}(2), 130-140. \doi{10.1177/0748730412475041}.

Roenneberg T., Allebrandt K. V., Merrow M., & Vetter C. (2012). Social jetlag
and obesity. \emph{Current Biology}, \emph{22}(10), 939-43.
\doi{10.1016/j.cub.2012.03.038}.

Roenneberg, T., Pilz, L. K., Zerbini, G., & Winnebeck, E. C. (2019).
Chronotype and social jetlag: a (self-) critical review. \emph{Biology}, \emph{8}(3),
54. \doi{10.3390/biology8030054}.

Roenneberg, T., Wirz-Justice, A., & Merrow, M. (2003). Life between clocks:
daily temporal patterns of human chronotypes. \emph{Journal of Biological
Rhythms}, \emph{18}(1), 80-90. \doi{10.1177/0748730402239679}.

The Worldwide Experimental Platform (n.d.). MCTQ.
\url{https://www.thewep.org/documentations/mctq/}
}
\seealso{
Other MCTQ functions: 
\code{\link{fd}()},
\code{\link{gu}()},
\code{\link{le_week}()},
\code{\link{msf_sc}()},
\code{\link{msl}()},
\code{\link{napd}()},
\code{\link{sd24}()},
\code{\link{sd_overall}()},
\code{\link{sd_week}()},
\code{\link{sdu}()},
\code{\link{sjl_sc}()},
\code{\link{sjl_weighted}()},
\code{\link{so}()},
\code{\link{tbt}()}
}
\concept{MCTQ functions}
