\name{cfunction}

\alias{cfunction}
\alias{setCMethod}
\alias{inline}

\concept{inline function call}

\title{ Inline C, C++, Fortran function calls from R }

\description{
  Functionality to dynamically define R functions and S4 methods with in-lined C, 
  C++ or Fortran code supporting .C and .Call calling conventions.
}

\usage{
  cfunction(sig=character(), body=character(), includes=character(), 
            language=c("C++", "C", "Fortran", "F95", "ObjectiveC", "ObjectiveC++"),
            verbose=FALSE, 
            convention=c(".Call", ".C", ".Fortran") )

  ## S4 methods for signatures
  #  f='character', sig='list', body='list'
  #  f='character', sig='character', body='character'
  
  setCMethod(f, sig, body, ...)

  ## Further arguments:
  #  setCMethod(f, sig, body, includes="", cpp=TRUE, verbose=FALSE, where=topenv(.GlobalEnv), ...)
}

\arguments{
  \item{f}{A single character value if \code{sig} and \code{body} are character vectors
    or a character vector of the same length and the length of \code{sig} or
    \code{body} with the name(s) of methods to create.}
    
  \item{sig}{A match of formal argument names for the function with the
    character-string names of corresponding classes. Alternatively, 
    a list of such character vectors. }

  \item{body}{ A character vector with C, C++ or Fortran code omitting function
    declaration (only the body, i.e. in case of C starting after the function 
    opening curly bracket and ending before the closing curly bracket, 
    brackets excluded). In case of \code{setCMethod} with signature
    \code{list} -- a list of such character vectors. } 

  \item{includes}{ A character vector of additional includes and preprocessor
    statements etc that will be put between the R includes and the user function(s).}

  \item{language}{ A character value that specifies the source language of the
    inline code, defaulting to C++.  In this case all functions are additionally
    listed in the \code{extern "C"} clause.}
    
  \item{verbose}{ If \code{TRUE} prints the compilation output, the source
    code of the resulting program and the definitions of all declared
    methods. If \code{FALSE}, the function is silent, but it prints compiler
    warning and error messages and the source code if compilation fails. }
    
  \item{convention}{ Which calling convention to use?  See the Details section.}

  \item{...}{ Reserved.}
}

\value{
  If \code{sig} is a single character vector, \code{cfunction} returns a single
  \code{\link{function}}; if it is a list, it returns a list of functions.
  
  \code{setCMethod} declares new methods with given names and signatures and
  returns invisible \code{NULL}.
}

\details{
  
  To declare multiple functions in the same library one can use \code{setCMethod}
  supplying lists of signatures and implementations. In this case, provide as
  many method names in \code{f} as you define methods. Avoid clashes when selecting
  names of the methods to declare, i.e. if you provide the same name several times
  you must ensure that signatures are different but can share the same generic!

  The source code in the \code{body} should not include the header or
  "front-matter" of the function or the close, e.g. in C or C++ it
  must start after the C-function opening curly bracket and end before
  the C-function closing curly bracket, brackets should not be
  included. The header will be automatically generated from the R-\code{signature} 
  argument. Arguments will will carry the same name as used in the signature, 
  so avoid variable names that are not legal in the target language 
  (e.g. names with dots).
  
  C/C++: If \code{convention == ".Call"} (the default), the \code{\link{.Call}} mechanism 
  is used and its result is returned directly as the result of the call of the 
  generated function.  As the last line of the generated C/C++ code a 
  \code{return R_NilValue;} is added in this case and a warning is generated 
  in case the user has forgotten to provide a return value. To suppress the 
  warning and still return NULL, add \code{return R_NilValue;} explicitly.
  
  Special care is needed with types, memory allocation and protection
  -- exactly the same as if the code was not inline: see the
  Writing R Extension manual for information on \code{\link{.Call}}.  
  
  If \code{convention == ".C"} or \code{convention == ".Fortran"}, the
  \code{\link{.C}} or \code{\link{.Fortran}} mechanism respectively is
  used, and the return value is a list containing all arguments.
  
  Attached R includes include \code{R.h} for \code{".C"}, and
  additionally \code{Rdefines.h} and \code{R_ext\\Error.h} for
  \code{".Call"}.
  
  The possible values for \code{language} include all languages supported
  by \code{R CMD SHLIB} on any platform; they may not all be supported
  on your platform. Please note that authors have not tested ObjectiveC and
  ObjectiveC++ due to the lack of corresponding use cases.
 
}

\seealso{ \code{
    \link{Foreign} Function Interface
}}

\examples{

## A simple Fortran example
code <- "
      integer i
      do 1 i=1, n(1)
    1 x(i) = x(i)**3
"
cubefn <- cfunction(signature(n="integer", x="double"), code, convention=".Fortran")

x <- 1:10
cubefn(10, x)$x

## Use of .C convention with C code
## Defining two functions, one of which calls the other
code <- c("
  for (int i=0; i < *n; i++) {
    x[i] = x[i]*x[i];
  }
", "
  squarefn(n, x);
  squarefn(n, x);
")

fns <- cfunction(list(squarefn=signature(n="integer", x="double"),
                      quadfn  =signature(n="integer", x="double")), 
                      code, convention=".C")
squarefn <- fns[["squarefn"]]
quadfn <- fns[["quadfn"]]

squarefn(10, x)$x
quadfn(10, x)$x

## Use of .Call convention with C code
## Multyplying each image in a stack with a 2D Gaussian at a given position
code <- "
  SEXP res;
  int nprotect = 0, nx, ny, nz, x, y;
  PROTECT(res = Rf_duplicate(a)); nprotect++;
  nx = INTEGER(GET_DIM(a))[0];
  ny = INTEGER(GET_DIM(a))[1];
  nz = INTEGER(GET_DIM(a))[2];
  double sigma2 = REAL(s)[0] * REAL(s)[0], d2 ;
  double cx = REAL(centre)[0], cy = REAL(centre)[1], *data, *rdata;
  for (int im = 0; im < nz; im++) {
    data = &(REAL(a)[im*nx*ny]); rdata = &(REAL(res)[im*nx*ny]);
    for (x = 0; x < nx; x++)
      for (y = 0; y < ny; y++) {
        d2 = (x-cx)*(x-cx) + (y-cy)*(y-cy);
        rdata[x + y*nx] = data[x + y*nx] * exp(-d2/sigma2);
      }
  }
  UNPROTECT(nprotect);
  return res;
"
funx <- cfunction(signature(a="array", s="numeric", centre="numeric"), code)

x <- array(runif(50*50), c(50,50,1))
res <- funx(a=x, s=10, centre=c(25,15))
if (interactive()) image(res[,,1])

## Same but done by registering an S4 method
setCMethod("funy", signature(a="array", s="numeric", centre="numeric"), code, verbose=TRUE)

res <- funy(x, 10, c(35,35))
if (interactive()) { x11(); image(res[,,1]) }

}

\author{
  Oleg Sklyar <\email{osklyar@ebi.ac.uk}>
  Duncan Murdoch
  Mike Smith
}

\keyword{file}

