% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Interaction.R
\name{Interaction}
\alias{Interaction}
\title{Feature interactions}
\format{\code{\link{R6Class}} object.}
\description{
\code{Interaction} estimates the feature interactions in a prediction model.
}
\section{Usage}{

\preformatted{
ia = Interaction$new(predictor, feature = NULL, grid.size = 20, run = TRUE)

plot(ia)
ia$results
print(ia)
}
}

\section{Arguments}{


For Interaction$new():
\describe{
\item{predictor: }{(Predictor)\cr 
The object (created with Predictor$new()) holding the machine learning model and the data.}
\item{feature: }{(`numeric(1)`|NULL)\cr 
If NULL, for each feature the interactions with all other features are estimated.
If one feature name is selected, the 2-way interactions of this feature with all other features are estimated}
\item{grid.size: }{(`logical(1)`)\cr The number of values per feature that should be used to estimate the interaction strength.
A larger grid.size means more accurate the results but longer the computation time.
For each of the grid points, the partial dependence functions have to be computed, which involves marginalizing over all data points.}
\item{run: }{(`logical(1)`)\cr Should the Interpretation method be run?}
}
}

\section{Details}{
  
Interactions between features are measured via the decomposition of the prediction function:
If a feature j has no interaction with any other feature, the prediction function can 
be expressed as the sum of the partial function that depends only on j and the partial function that only depends on features other than j.
If the variance of the full function is completely explained by the sum of the partial functions, there is no interaction between feature j and the other features. 
Any variance that is not explained can be attributed to the interaction and is used as a measure of interaction strength.

The interaction strength between two features is the proportion of the variance of the 2-dimensional partial dependence function that is not explained
by the sum of the two 1-dimensional partial dependence functions.

The interaction measure takes on values between 0 (no interaction) to 1 (100% of variance of f(x) du to interaction).
}

\section{Fields}{

\describe{
\item{grid.size: }{(`logical(1)`)\cr The number of values per feature that should be used to estimate the interaction strength.}
\item{predictor: }{(Predictor)\cr The prediction model that was analysed.}
\item{results: }{(data.frame)\cr Data.frame with the interaction strength (column '.interation') per feature and - in the case of a multi-dimensional outcome - per class.}
}
}

\section{Methods}{

\describe{
\item{plot()}{method to plot the feature interactions. See \link{plot.Interaction}.}
\item{\code{run()}}{[internal] method to run the interpretability method. Use \code{obj$run(force = TRUE)} to force a rerun.}
\item{\code{clone()}}{[internal] method to clone the R6 object.}
\item{\code{initialize()}}{[internal] method to initialize the R6 object.}
}
}

\examples{
if (require("rpart")) {
set.seed(42)
# Fit a CART on the Boston housing data set
data("Boston", package  = "MASS")
rf = rpart(medv ~ ., data = Boston)
# Create a model object
mod = Predictor$new(rf, data = Boston[-which(names(Boston) == "medv")]) 

# Measure the interaction strength
ia = Interaction$new(mod)

# Plot the resulting leaf nodes
plot(ia) 


# Extract the results
dat = ia$results
head(dat)
\dontrun{
# Interaction also works with multiclass classification
rf = rpart(Species ~ ., data = iris)
predict.fun = function(object, newdata) predict(object, newdata, type = "prob")
mod = Predictor$new(rf, data = iris, predict.fun = predict.fun)

# For some models we have to specify additional arguments for the predict function
ia = Interaction$new(mod)

ia$plot()

# For multiclass classification models, you can choose to only show one class:
mod = Predictor$new(rf, data = iris, predict.fun = predict.fun, class = "virginica")
plot(Interaction$new(mod))
}
}
}
\references{
Friedman, Jerome H., and Bogdan E. Popescu. "Predictive learning via rule ensembles." 
The Annals of Applied Statistics 2.3 (2008): 916-954.
}
