#' Converts the HTTK-Pop population data table to a table of the parameters
#' needed by HTTK, for a specific chemical.
#' 
#' Takes the data table generated by \code{\link{httkpop_generate}}, and converts it
#' to the corresponding table of HTTK model parameters for a specified chemical
#' and HTTK model.
#' 
#' 
#' @param model One of the HTTK models: "1compartment", "3compartmentss",
#' "3compartment", or "pbtk".
#' @param samples The number of Monte Carlo samples to use (can often think of these
#' as separate individuals)
#' @param httkpop.dt A data table generated by \code{\link{httkpop_generate}}.
#' This defaults to NULL, in which case \code{\link{httkpop_generate}} is 
#' called to generate this table.
#' @param ... Additional arugments passed on to \code{\link{httkpop_generate}}.
#' 
#' @author Caroline Ring and John Wambaugh
#' 
#' @return 
#' A data.table with a row for each individual in the sample and a column for
#' each parater in the model.
#' 
#' @references Ring, Caroline L., et al. "Identifying populations sensitive to
#' environmental chemicals by simulating toxicokinetic variability."
#' Environment International 106 (2017): 105-118
#' 
#' Rowland, Malcolm, Leslie Z. Benet, and Garry G. Graham. "Clearance concepts
#' in pharmacokinetics." Journal of Pharmacokinetics and Biopharmaceutics 1.2
#' (1973): 123-136. 
#'
#' @examples
#' 
#' set.seed(42)
#' indiv_examp <- httkpop_generate(method="d", nsamp=100)
#' httk_param <- httkpop_mc(httkpop.dt=indiv_examp, 
#' model="1compartment")
#' 
#' @keywords httk-pop monte-carlo
#'
#' @export httkpop_mc
httkpop_mc <- function(model,
                       samples=1000,
                       httkpop.dt=NULL,
                       ...)
{
  if (is.null(model)) stop("Model must be specified.")
# We need to know model-specific information (from modelinfo_[MODEL].R]) 
# to set up the solver:
  model <- tolower(model)
  if (!(model %in% names(model.list)))            
  {
    stop(paste("Model",model,"not available. Please select from:",
      paste(names(model.list),collapse=", ")))
  }

# Generate the initial physiology data from NHANES biometrics:
  if (is.null(httkpop.dt))
    httkpop.dt <- do.call(
                        httkpop_generate,
                        args=c(list(nsamp=samples),
                          ...))

  # Convert HTTK-Pop-generated parameters to HTTK physiological parameters
  if (model.list[[model]]$calc.standard.httkpop2httk)
    physiology.dt <- httkpop_biotophys_default(indiv_dt = httkpop.dt)
  

  return(physiology.dt)
}
