% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/hdnom.calibrate.R
\name{hdnom.calibrate}
\alias{hdnom.calibrate}
\title{Calibrate High-Dimensional Cox models}
\usage{
hdnom.calibrate(x, time, event, alpha, lambda, pen.factor = NULL,
  method = c("fitting", "bootstrap", "cv", "repeated.cv"),
  boot.times = NULL, nfolds = NULL, rep.times = NULL, pred.at,
  ngroup = 5, trace = TRUE)
}
\arguments{
\item{x}{Matrix of training data used for the \code{glmnet} object;
on which to run the calibration.}

\item{time}{Survival time.
Must be of the same length with the number of rows as \code{x}.}

\item{event}{Status indicator, normally 0 = alive, 1 = dead.
Must be of the same length with the number of rows as \code{x}.}

\item{alpha}{Value of the elastic-net mixing parameter alpha in
glmnet. \code{alpha = 1}: lasso; \code{alpha = 0}: ridge.
From the Cox model you have built.}

\item{lambda}{Value of the penalty parameter lambda to use in the
glmnet fits on the resampled data. From the Cox model you have built.}

\item{pen.factor}{Penalty factors to apply to each coefficient.
From the \emph{adaptive} penalized Cox model you have built.}

\item{method}{Calibration method.
Options including \code{"fitting"}, \code{"bootstrap"}, \code{"cv"},
and \code{"repeated.cv"}.}

\item{boot.times}{Number of repetitions for bootstrap.}

\item{nfolds}{Number of folds for cross-validation and
repeated cross-validation.}

\item{rep.times}{Number of repeated times for repeated cross-validation.}

\item{pred.at}{Time point at which calibration should take place.}

\item{ngroup}{Number of groups to be formed for calibration.}

\item{trace}{Logical. Print trace or not. Default is \code{TRUE}.}
}
\description{
Calibrate High-Dimensional Cox models
}
\examples{
library("glmnet")
library("survival")
library("rms")

# Load imputed SMART data
data(smart)
x = as.matrix(smart[, -c(1, 2)])
time = smart$TEVENT
event = smart$EVENT

# Fit penalized Cox model (lasso penalty) with glmnet
set.seed(1010)
cvfit = cv.glmnet(x, Surv(time, event), family = "cox", nfolds = 5)

# Model calibration by fitting the original data directly
cal.fitting = hdnom.calibrate(x, time, event,
                              alpha = 1, lambda = cvfit$lambda.1se,
                              method = "fitting",
                              pred.at = 365 * 9, ngroup = 5)

# Model calibration by bootstrap
# Normally boot.times should be set to 200 or more,
# we set it to 3 here only to save example running time.
cal.boot = hdnom.calibrate(x, time, event,
                           alpha = 1, lambda = cvfit$lambda.1se,
                           method = "bootstrap", boot.times = 3,
                           pred.at = 365 * 9, ngroup = 5)

# Model calibration by 10-fold cross-validation
cal.cv = hdnom.calibrate(x, time, event,
                         alpha = 1, lambda = cvfit$lambda.1se,
                         method = "cv", nfolds = 5,
                         pred.at = 365 * 9, ngroup = 5)

# Model calibration by repeated cross-validation
cal.repcv = hdnom.calibrate(x, time, event,
                            alpha = 1, lambda = cvfit$lambda.1se,
                            method = "repeated.cv", nfolds = 5, rep.times = 3,
                            pred.at = 365 * 9, ngroup = 5)

print(cal.fitting)
summary(cal.fitting)
plot(cal.fitting)

print(cal.boot)
summary(cal.boot)
plot(cal.boot)

print(cal.cv)
summary(cal.cv)
plot(cal.cv)

print(cal.repcv)
summary(cal.repcv)
plot(cal.repcv)
}

