/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5VMprivate.h" 

static herr_t H5D__select_io(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info,
                             size_t elmt_size);

H5FL_SEQ_DEFINE(size_t);

H5FL_SEQ_DEFINE(hsize_t);

H5FL_EXTERN(H5S_sel_iter_t);

static herr_t
H5D__select_io(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info, size_t elmt_size)
{
    H5S_sel_iter_t *mem_iter       = NULL;  
    bool            mem_iter_init  = false; 
    H5S_sel_iter_t *file_iter      = NULL;  
    bool            file_iter_init = false; 
    hsize_t        *mem_off        = NULL;  
    hsize_t        *file_off       = NULL;  
    size_t         *mem_len        = NULL;  
    size_t         *file_len       = NULL;  
    size_t          curr_mem_seq;           
    size_t          curr_file_seq;          
    size_t          mem_nseq;               
    size_t          file_nseq;              
    size_t          dxpl_vec_size;          
    size_t          vec_size;               
    ssize_t         tmp_file_len;           
    size_t          nelmts;                 
    herr_t          ret_value = SUCCEED;    

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info->dset);
    assert(dset_info->store);
    assert(dset_info->buf.vp);

    if (elmt_size == 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_BADVALUE, FAIL, "invalid elmt_size of 0");

    
    nelmts = dset_info->nelmts;

    
    if (nelmts == 1) {
        hsize_t single_mem_off;  
        hsize_t single_file_off; 
        size_t  single_mem_len;  
        size_t  single_file_len; 

        
        if (H5S_SELECT_OFFSET(dset_info->file_space, &single_file_off) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, FAIL, "can't retrieve file selection offset");
        if (H5S_SELECT_OFFSET(dset_info->mem_space, &single_mem_off) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, FAIL, "can't retrieve memory selection offset");

        
        file_nseq = mem_nseq = 1;
        curr_mem_seq = curr_file_seq = 0;
        single_file_off *= elmt_size;
        single_mem_off *= elmt_size;
        single_file_len = single_mem_len = elmt_size;

        
        if (io_info->op_type == H5D_IO_OP_READ) {
            if ((tmp_file_len = (*dset_info->layout_ops.readvv)(
                     io_info, dset_info, file_nseq, &curr_file_seq, &single_file_len, &single_file_off,
                     mem_nseq, &curr_mem_seq, &single_mem_len, &single_mem_off)) < 0)
                HGOTO_ERROR(H5E_DATASPACE, H5E_READERROR, FAIL, "read error");
        } 
        else {
            assert(io_info->op_type == H5D_IO_OP_WRITE);
            if ((tmp_file_len = (*dset_info->layout_ops.writevv)(
                     io_info, dset_info, file_nseq, &curr_file_seq, &single_file_len, &single_file_off,
                     mem_nseq, &curr_mem_seq, &single_mem_len, &single_mem_off)) < 0)
                HGOTO_ERROR(H5E_DATASPACE, H5E_WRITEERROR, FAIL, "write error");
        } 

        
        assert(((size_t)tmp_file_len % elmt_size) == 0);
    } 
    else {
        size_t mem_nelem;  
        size_t file_nelem; 

        
        if (H5CX_get_vec_size(&dxpl_vec_size) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve I/O vector size");

        
        if (dxpl_vec_size > H5D_IO_VECTOR_SIZE)
            vec_size = dxpl_vec_size;
        else
            vec_size = H5D_IO_VECTOR_SIZE;
        if (NULL == (mem_len = H5FL_SEQ_MALLOC(size_t, vec_size)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate I/O length vector array");
        if (NULL == (mem_off = H5FL_SEQ_MALLOC(hsize_t, vec_size)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate I/O offset vector array");
        if (NULL == (file_len = H5FL_SEQ_MALLOC(size_t, vec_size)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate I/O length vector array");
        if (NULL == (file_off = H5FL_SEQ_MALLOC(hsize_t, vec_size)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate I/O offset vector array");

        
        if (NULL == (mem_iter = H5FL_MALLOC(H5S_sel_iter_t)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate memory iterator");
        if (NULL == (file_iter = H5FL_MALLOC(H5S_sel_iter_t)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate file iterator");

        
        if (H5S_select_iter_init(file_iter, dset_info->file_space, elmt_size,
                                 H5S_SEL_ITER_GET_SEQ_LIST_SORTED) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINIT, FAIL, "unable to initialize selection iterator");
        file_iter_init = 1; 

        
        if (H5S_select_iter_init(mem_iter, dset_info->mem_space, elmt_size, 0) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINIT, FAIL, "unable to initialize selection iterator");
        mem_iter_init = 1; 

        
        curr_mem_seq = curr_file_seq = 0;
        mem_nseq = file_nseq = 0;

        
        while (nelmts > 0) {
            
            if (curr_file_seq >= file_nseq) {
                
                if (H5S_SELECT_ITER_GET_SEQ_LIST(file_iter, vec_size, nelmts, &file_nseq, &file_nelem,
                                                 file_off, file_len) < 0)
                    HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, FAIL, "sequence length generation failed");

                
                curr_file_seq = 0;
            } 

            
            if (curr_mem_seq >= mem_nseq) {
                
                if (H5S_SELECT_ITER_GET_SEQ_LIST(mem_iter, vec_size, nelmts, &mem_nseq, &mem_nelem, mem_off,
                                                 mem_len) < 0)
                    HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, FAIL, "sequence length generation failed");

                
                curr_mem_seq = 0;
            } 

            
            if (io_info->op_type == H5D_IO_OP_READ) {
                if ((tmp_file_len = (*dset_info->layout_ops.readvv)(
                         io_info, dset_info, file_nseq, &curr_file_seq, file_len, file_off, mem_nseq,
                         &curr_mem_seq, mem_len, mem_off)) < 0)
                    HGOTO_ERROR(H5E_DATASPACE, H5E_READERROR, FAIL, "read error");
            } 
            else {
                assert(io_info->op_type == H5D_IO_OP_WRITE);
                if ((tmp_file_len = (*dset_info->layout_ops.writevv)(
                         io_info, dset_info, file_nseq, &curr_file_seq, file_len, file_off, mem_nseq,
                         &curr_mem_seq, mem_len, mem_off)) < 0)
                    HGOTO_ERROR(H5E_DATASPACE, H5E_WRITEERROR, FAIL, "write error");
            } 

            
            assert(((size_t)tmp_file_len % elmt_size) == 0);
            nelmts -= ((size_t)tmp_file_len / elmt_size);
        } 
    }     

done:
    
    if (file_iter_init && H5S_SELECT_ITER_RELEASE(file_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to release selection iterator");
    if (file_iter)
        file_iter = H5FL_FREE(H5S_sel_iter_t, file_iter);
    if (mem_iter_init && H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to release selection iterator");
    if (mem_iter)
        mem_iter = H5FL_FREE(H5S_sel_iter_t, mem_iter);

    
    if (file_len)
        file_len = H5FL_SEQ_FREE(size_t, file_len);
    if (file_off)
        file_off = H5FL_SEQ_FREE(hsize_t, file_off);
    if (mem_len)
        mem_len = H5FL_SEQ_FREE(size_t, mem_len);
    if (mem_off)
        mem_off = H5FL_SEQ_FREE(hsize_t, mem_off);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D_select_io_mem(void *dst_buf, H5S_t *dst_space, const void *src_buf, H5S_t *src_space, size_t elmt_size,
                  size_t nelmts)
{
    H5S_sel_iter_t *dst_sel_iter      = NULL;  
    H5S_sel_iter_t *src_sel_iter      = NULL;  
    bool            dst_sel_iter_init = false; 
    bool            src_sel_iter_init = false; 
    hsize_t        *dst_off           = NULL;  
    hsize_t        *src_off           = NULL;  
    size_t         *dst_len           = NULL;  
    size_t         *src_len           = NULL;  
    size_t          curr_dst_seq;              
    size_t          curr_src_seq;              
    size_t          dst_nseq;                  
    size_t          src_nseq;                  
    size_t          dxpl_vec_size;             
    size_t          vec_size;                  
    ssize_t         bytes_copied;
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    assert(dst_buf);
    assert(dst_space);
    assert(src_buf);
    assert(src_space);

    if (elmt_size == 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_BADVALUE, FAIL, "invalid elmt_size of 0");

    
    if (nelmts == 1) {
        hsize_t single_dst_off; 
        hsize_t single_src_off; 
        size_t  single_dst_len; 
        size_t  single_src_len; 

        
        if (H5S_SELECT_OFFSET(dst_space, &single_dst_off) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "can't retrieve destination selection offset");
        if (H5S_SELECT_OFFSET(src_space, &single_src_off) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "can't retrieve source selection offset");

        
        dst_nseq = src_nseq = 1;
        curr_dst_seq = curr_src_seq = 0;
        single_dst_off *= elmt_size;
        single_src_off *= elmt_size;
        single_dst_len = single_src_len = elmt_size;

        
        if ((bytes_copied =
                 H5VM_memcpyvv(dst_buf, dst_nseq, &curr_dst_seq, &single_dst_len, &single_dst_off, src_buf,
                               src_nseq, &curr_src_seq, &single_src_len, &single_src_off)) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "vectorized memcpy failed");

        assert(((size_t)bytes_copied % elmt_size) == 0);
    }
    else {
        unsigned sel_iter_flags = H5S_SEL_ITER_GET_SEQ_LIST_SORTED | H5S_SEL_ITER_SHARE_WITH_DATASPACE;
        size_t   dst_nelem; 
        size_t   src_nelem; 

        
        if (H5CX_get_vec_size(&dxpl_vec_size) < 0)
            HGOTO_ERROR(H5E_IO, H5E_CANTGET, FAIL, "can't retrieve I/O vector size");

        
        if (dxpl_vec_size > H5D_IO_VECTOR_SIZE)
            vec_size = dxpl_vec_size;
        else
            vec_size = H5D_IO_VECTOR_SIZE;

        if (NULL == (dst_len = H5FL_SEQ_MALLOC(size_t, vec_size)))
            HGOTO_ERROR(H5E_IO, H5E_CANTALLOC, FAIL, "can't allocate I/O length vector array");
        if (NULL == (dst_off = H5FL_SEQ_MALLOC(hsize_t, vec_size)))
            HGOTO_ERROR(H5E_IO, H5E_CANTALLOC, FAIL, "can't allocate I/O offset vector array");
        if (NULL == (src_len = H5FL_SEQ_MALLOC(size_t, vec_size)))
            HGOTO_ERROR(H5E_IO, H5E_CANTALLOC, FAIL, "can't allocate I/O length vector array");
        if (NULL == (src_off = H5FL_SEQ_MALLOC(hsize_t, vec_size)))
            HGOTO_ERROR(H5E_IO, H5E_CANTALLOC, FAIL, "can't allocate I/O offset vector array");

        
        if (NULL == (dst_sel_iter = H5FL_MALLOC(H5S_sel_iter_t)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTALLOC, FAIL, "can't allocate destination selection iterator");
        if (NULL == (src_sel_iter = H5FL_MALLOC(H5S_sel_iter_t)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTALLOC, FAIL, "can't allocate source selection iterator");

        
        if (H5S_select_iter_init(dst_sel_iter, dst_space, elmt_size, sel_iter_flags) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINIT, FAIL, "unable to initialize selection iterator");
        dst_sel_iter_init = true; 

        
        if (H5S_select_iter_init(src_sel_iter, src_space, elmt_size, H5S_SEL_ITER_SHARE_WITH_DATASPACE) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINIT, FAIL, "unable to initialize selection iterator");
        src_sel_iter_init = true; 

        
        curr_dst_seq = curr_src_seq = 0;
        dst_nseq = src_nseq = 0;

        
        while (nelmts > 0) {
            
            if (curr_dst_seq >= dst_nseq) {
                
                if (H5S_SELECT_ITER_GET_SEQ_LIST(dst_sel_iter, vec_size, nelmts, &dst_nseq, &dst_nelem,
                                                 dst_off, dst_len) < 0)
                    HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "sequence length generation failed");

                
                curr_dst_seq = 0;
            }

            
            if (curr_src_seq >= src_nseq) {
                
                if (H5S_SELECT_ITER_GET_SEQ_LIST(src_sel_iter, vec_size, nelmts, &src_nseq, &src_nelem,
                                                 src_off, src_len) < 0)
                    HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "sequence length generation failed");

                
                curr_src_seq = 0;
            } 

            
            if ((bytes_copied = H5VM_memcpyvv(dst_buf, dst_nseq, &curr_dst_seq, dst_len, dst_off, src_buf,
                                              src_nseq, &curr_src_seq, src_len, src_off)) < 0)
                HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "vectorized memcpy failed");

            
            assert(((size_t)bytes_copied % elmt_size) == 0);
            nelmts -= ((size_t)bytes_copied / elmt_size);
        }
    }

done:
    
    if (src_sel_iter) {
        if (src_sel_iter_init && H5S_SELECT_ITER_RELEASE(src_sel_iter) < 0)
            HDONE_ERROR(H5E_DATASPACE, H5E_CANTRELEASE, FAIL, "unable to release selection iterator");

        src_sel_iter = H5FL_FREE(H5S_sel_iter_t, src_sel_iter);
    }
    if (dst_sel_iter) {
        if (dst_sel_iter_init && H5S_SELECT_ITER_RELEASE(dst_sel_iter) < 0)
            HDONE_ERROR(H5E_DATASPACE, H5E_CANTRELEASE, FAIL, "unable to release selection iterator");

        dst_sel_iter = H5FL_FREE(H5S_sel_iter_t, dst_sel_iter);
    }

    
    if (src_off)
        src_off = H5FL_SEQ_FREE(hsize_t, src_off);
    if (src_len)
        src_len = H5FL_SEQ_FREE(size_t, src_len);
    if (dst_off)
        dst_off = H5FL_SEQ_FREE(hsize_t, dst_off);
    if (dst_len)
        dst_len = H5FL_SEQ_FREE(size_t, dst_len);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__select_read(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5D__select_io(io_info, dset_info, dset_info->type_info.src_type_size) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_READERROR, FAIL, "read error");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__select_write(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5D__select_io(io_info, dset_info, dset_info->type_info.dst_type_size) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_WRITEERROR, FAIL, "write error");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
