% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_datetime}
\alias{fmt_datetime}
\title{Format values as datetimes}
\usage{
fmt_datetime(
  data,
  columns = everything(),
  rows = everything(),
  date_style = "iso",
  time_style = "iso",
  sep = " ",
  format = NULL,
  tz = NULL,
  pattern = "{x}",
  locale = NULL
)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{columns}{\emph{Columns to target}

\verb{<column-targeting expression>} // \emph{default:} \code{everything()}

Can either be a series of column names provided in \code{c()}, a vector of
column indices, or a select helper function (e.g. \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}} and \code{\link[=everything]{everything()}}).}

\item{rows}{\emph{Rows to target}

\verb{<row-targeting expression>} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
undergo formatting. The default \code{\link[=everything]{everything()}} results in all rows in
\code{columns} being formatted. Alternatively, we can supply a vector of row
captions within \code{c()}, a vector of row indices, or a select helper
function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{date_style}{\emph{Predefined style for dates}

\verb{scalar<character>|scalar<numeric|integer>(1<=val<=41)} // \emph{default:} \code{"iso"}

The date style to use. By default this is the short name \code{"iso"} which
corresponds to ISO 8601 date formatting. There are 41 date styles in total
and their short names can be viewed using \code{\link[=info_date_style]{info_date_style()}}.}

\item{time_style}{\emph{Predefined style for times}

\verb{scalar<character>|scalar<numeric|integer>(1<=val<=25)} // \emph{default:} \code{"iso"}

The time style to use. By default this is the short name \code{"iso"} which
corresponds to how times are formatted within ISO 8601 datetime values.
There are 25 time styles in total and their short names can be viewed using
\code{\link[=info_time_style]{info_time_style()}}.}

\item{sep}{\emph{Separator between date and time components}

\verb{scalar<character>} // \emph{default:} \code{" "}

The separator string to use between the date and time components. By
default, this is a single space character (\code{" "}). Only used when not
specifying a \code{format} code.}

\item{format}{\emph{Date/time formatting string}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

An optional formatting string used for generating custom dates/times. If
used then the arguments governing preset styles (\code{date_style} and
\code{time_style}) will be ignored in favor of formatting via the \code{format}
string.}

\item{tz}{\emph{Time zone}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

The time zone for printing dates/times (i.e., the output). The
default of \code{NULL} will preserve the time zone of the input data in the
output. If providing a time zone, it must be one that is recognized by the
user's operating system (a vector of all valid \code{tz} values can be produced
with \code{\link[=OlsonNames]{OlsonNames()}}).}

\item{pattern}{\emph{Specification of the formatting pattern}

\verb{scalar<character>} // \emph{default:} \code{"{x}"}

A formatting pattern that allows for decoration of the formatted value. The
formatted value is represented by the \code{{x}} (which can be used multiple
times, if needed) and all other characters will be interpreted as string
literals.}

\item{locale}{\emph{Locale identifier}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

An optional locale identifier that can be used for formatting values
according the locale's rules. Examples include \code{"en"} for English (United
States) and \code{"fr"} for French (France). We can call \code{\link[=info_locales]{info_locales()}} for a
useful reference for all of the locales that are supported. A locale ID can
be also set in the initial \code{\link[=gt]{gt()}} function call (where it would be used
automatically by any function with a \code{locale} argument) but a \code{locale}
value provided here will override that global locale.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Format input values to datetime values using either presets for the date and
time components or a formatting directive (this can either use a \emph{CLDR}
datetime pattern or \code{strptime} formatting). The input values can be in the
form of \code{POSIXct} (i.e., datetimes), the \code{Date} type, or \code{character} (must be
in the ISO 8601 form of \verb{YYYY-MM-DD HH:MM:SS} or \code{YYYY-MM-DD}).
}
\section{Compatibility of formatting function with data values}{


\code{fmt_datetime()} is compatible with body cells that are of the \code{"Date"},
\code{"POSIXct"} or \code{"character"} types. Any other types of body cells are ignored
during formatting. This is to say that cells of incompatible data types may
be targeted, but there will be no attempt to format them.
}

\section{Targeting cells with \code{columns} and \code{rows}}{


Targeting of values is done through \code{columns} and additionally by \code{rows} (if
nothing is provided for \code{rows} then entire columns are selected). The
\code{columns} argument allows us to target a subset of cells contained in the
resolved columns. We say resolved because aside from declaring column names
in \code{c()} (with bare column names or names in quotes) we can use
\strong{tidyselect}-style expressions. This can be as basic as supplying a select
helper like \code{starts_with()}, or, providing a more complex incantation like

\code{where(~ is.numeric(.x) && max(.x, na.rm = TRUE) > 1E6)}

which targets numeric columns that have a maximum value greater than
1,000,000 (excluding any \code{NA}s from consideration).

By default all columns and rows are selected (with the \code{everything()}
defaults). Cell values that are incompatible with a given formatting function
will be skipped over, like \code{character} values and numeric \verb{fmt_*()}
functions. So it's safe to select all columns with a particular formatting
function (only those values that can be formatted will be formatted), but,
you may not want that. One strategy is to format the bulk of cell values with
one formatting function and then constrain the columns for later passes with
other types of formatting (the last formatting done to a cell is what you get
in the final output).

Once the columns are targeted, we may also target the \code{rows} within those
columns. This can be done in a variety of ways. If a stub is present, then we
potentially have row identifiers. Those can be used much like column names in
the \code{columns}-targeting scenario. We can use simpler \strong{tidyselect}-style
expressions (the select helpers should work well here) and we can use quoted
row identifiers in \code{c()}. It's also possible to use row indices (e.g.,
\code{c(3, 5, 6)}) though these index values must correspond to the row numbers of
the input data (the indices won't necessarily match those of rearranged rows
if row groups are present). One more type of expression is possible, an
expression that takes column values (can involve any of the available columns
in the table) and returns a logical vector. This is nice if you want to base
formatting on values in the column or another column, or, you'd like to use a
more complex predicate expression.
}

\section{Compatibility of arguments with the \code{from_column()} helper function}{


\code{\link[=from_column]{from_column()}} can be used with certain arguments of \code{fmt_datetime()} to
obtain varying parameter values from a specified column within the table.
This means that each row could be formatted a little bit differently. These
arguments provide support for \code{\link[=from_column]{from_column()}}:
\itemize{
\item \code{date_style}
\item \code{time_style}
\item \code{sep}
\item \code{format}
\item \code{tz}
\item \code{pattern}
\item \code{locale}
}

Please note that for each of the aforementioned arguments, a \code{\link[=from_column]{from_column()}}
call needs to reference a column that has data of the correct type (this is
different for each argument). Additional columns for parameter values can be
generated with \code{\link[=cols_add]{cols_add()}} (if not already present). Columns that contain
parameter data can also be hidden from final display with \code{\link[=cols_hide]{cols_hide()}}.
Finally, there is no limitation to how many arguments the \code{\link[=from_column]{from_column()}}
helper is applied so long as the arguments belong to this closed set.
}

\section{Formatting with the \code{date_style} argument}{


We can supply a preset date style to the \code{date_style} argument to separately
handle the date portion of the output. The date styles are numerous and can
handle localization to any supported locale. A large segment of date styles
are termed flexible date formats and this means that their output will adapt
to any \code{locale} provided. That feature makes the flexible date formats a
better option for locales other than \code{"en"} (the default locale).

The following table provides a listing of all date styles and their output
values (corresponding to an input date of \code{2000-02-29}).\tabular{llll}{
    \tab Date Style \tab Output \tab Notes \cr
   1 \tab \code{"iso"} \tab \code{"2000-02-29"} \tab ISO 8601 \cr
   2 \tab \code{"wday_month_day_year"} \tab \code{"Tuesday, February 29, 2000"} \tab  \cr
   3 \tab \code{"wd_m_day_year"} \tab \code{"Tue, Feb 29, 2000"} \tab  \cr
   4 \tab \code{"wday_day_month_year"} \tab \code{"Tuesday 29 February 2000"} \tab  \cr
   5 \tab \code{"month_day_year"} \tab \code{"February 29, 2000"} \tab  \cr
   6 \tab \code{"m_day_year"} \tab \code{"Feb 29, 2000"} \tab  \cr
   7 \tab \code{"day_m_year"} \tab \code{"29 Feb 2000"} \tab  \cr
   8 \tab \code{"day_month_year"} \tab \code{"29 February 2000"} \tab  \cr
   9 \tab \code{"day_month"} \tab \code{"29 February"} \tab  \cr
   10 \tab \code{"day_m"} \tab \code{"29 Feb"} \tab  \cr
   11 \tab \code{"year"} \tab \code{"2000"} \tab  \cr
   12 \tab \code{"month"} \tab \code{"February"} \tab  \cr
   13 \tab \code{"day"} \tab \code{"29"} \tab  \cr
   14 \tab \code{"year.mn.day"} \tab \code{"2000/02/29"} \tab  \cr
   15 \tab \code{"y.mn.day"} \tab \code{"00/02/29"} \tab  \cr
   16 \tab \code{"year_week"} \tab \code{"2000-W09"} \tab  \cr
   17 \tab \code{"year_quarter"} \tab \code{"2000-Q1"} \tab  \cr
   18 \tab \code{"yMd"} \tab \code{"2/29/2000"} \tab flexible \cr
   19 \tab \code{"yMEd"} \tab \code{"Tue, 2/29/2000"} \tab flexible \cr
   20 \tab \code{"yMMM"} \tab \code{"Feb 2000"} \tab flexible \cr
   21 \tab \code{"yMMMM"} \tab \code{"February 2000"} \tab flexible \cr
   22 \tab \code{"yMMMd"} \tab \code{"Feb 29, 2000"} \tab flexible \cr
   23 \tab \code{"yMMMEd"} \tab \code{"Tue, Feb 29, 2000"} \tab flexible \cr
   24 \tab \code{"GyMd"} \tab \code{"2/29/2000 A"} \tab flexible \cr
   25 \tab \code{"GyMMMd"} \tab \code{"Feb 29, 2000 AD"} \tab flexible \cr
   26 \tab \code{"GyMMMEd"} \tab \code{"Tue, Feb 29, 2000 AD"} \tab flexible \cr
   27 \tab \code{"yM"} \tab \code{"2/2000"} \tab flexible \cr
   28 \tab \code{"Md"} \tab \code{"2/29"} \tab flexible \cr
   29 \tab \code{"MEd"} \tab \code{"Tue, 2/29"} \tab flexible \cr
   30 \tab \code{"MMMd"} \tab \code{"Feb 29"} \tab flexible \cr
   31 \tab \code{"MMMEd"} \tab \code{"Tue, Feb 29"} \tab flexible \cr
   32 \tab \code{"MMMMd"} \tab \code{"February 29"} \tab flexible \cr
   33 \tab \code{"GyMMM"} \tab \code{"Feb 2000 AD"} \tab flexible \cr
   34 \tab \code{"yQQQ"} \tab \code{"Q1 2000"} \tab flexible \cr
   35 \tab \code{"yQQQQ"} \tab \code{"1st quarter 2000"} \tab flexible \cr
   36 \tab \code{"Gy"} \tab \code{"2000 AD"} \tab flexible \cr
   37 \tab \code{"y"} \tab \code{"2000"} \tab flexible \cr
   38 \tab \code{"M"} \tab \code{"2"} \tab flexible \cr
   39 \tab \code{"MMM"} \tab \code{"Feb"} \tab flexible \cr
   40 \tab \code{"d"} \tab \code{"29"} \tab flexible \cr
   41 \tab \code{"Ed"} \tab \code{"29 Tue"} \tab flexible \cr
}


We can call \code{\link[=info_date_style]{info_date_style()}} in the console to view a similar table of
date styles with example output.
}

\section{Formatting with the \code{time_style} argument}{


We can supply a preset time style to the \code{time_style} argument to separately
handle the time portion of the output. There are many time styles and all of
them can handle localization to any supported locale. Many of the time styles
are termed flexible time formats and this means that their output will adapt
to any \code{locale} provided. That feature makes the flexible time formats a
better option for locales other than \code{"en"} (the default locale).

The following table provides a listing of all time styles and their output
values (corresponding to an input time of \code{14:35:00}). It is noted which of
these represent 12- or 24-hour time. Some of the flexible formats (those
that begin with \code{"E"}) include the day of the week. Keep this in mind
when pairing such \code{time_style} values with a \code{date_style} so as to avoid
redundant or repeating information.\tabular{llll}{
    \tab Time Style \tab Output \tab Notes \cr
   1 \tab \code{"iso"} \tab \code{"14:35:00"} \tab ISO 8601, 24h \cr
   2 \tab \code{"iso-short"} \tab \code{"14:35"} \tab ISO 8601, 24h \cr
   3 \tab \code{"h_m_s_p"} \tab \code{"2:35:00 PM"} \tab 12h \cr
   4 \tab \code{"h_m_p"} \tab \code{"2:35 PM"} \tab 12h \cr
   5 \tab \code{"h_p"} \tab \code{"2 PM"} \tab 12h \cr
   6 \tab \code{"Hms"} \tab \code{"14:35:00"} \tab flexible, 24h \cr
   7 \tab \code{"Hm"} \tab \code{"14:35"} \tab flexible, 24h \cr
   8 \tab \code{"H"} \tab \code{"14"} \tab flexible, 24h \cr
   9 \tab \code{"EHm"} \tab \code{"Thu 14:35"} \tab flexible, 24h \cr
   10 \tab \code{"EHms"} \tab \code{"Thu 14:35:00"} \tab flexible, 24h \cr
   11 \tab \code{"Hmsv"} \tab \code{"14:35:00 GMT+00:00"} \tab flexible, 24h \cr
   12 \tab \code{"Hmv"} \tab \code{"14:35 GMT+00:00"} \tab flexible, 24h \cr
   13 \tab \code{"hms"} \tab \code{"2:35:00 PM"} \tab flexible, 12h \cr
   14 \tab \code{"hm"} \tab \code{"2:35 PM"} \tab flexible, 12h \cr
   15 \tab \code{"h"} \tab \code{"2 PM"} \tab flexible, 12h \cr
   16 \tab \code{"Ehm"} \tab \code{"Thu 2:35 PM"} \tab flexible, 12h \cr
   17 \tab \code{"Ehms"} \tab \code{"Thu 2:35:00 PM"} \tab flexible, 12h \cr
   18 \tab \code{"EBhms"} \tab \code{"Thu 2:35:00 in the afternoon"} \tab flexible, 12h \cr
   19 \tab \code{"Bhms"} \tab \code{"2:35:00 in the afternoon"} \tab flexible, 12h \cr
   20 \tab \code{"EBhm"} \tab \code{"Thu 2:35 in the afternoon"} \tab flexible, 12h \cr
   21 \tab \code{"Bhm"} \tab \code{"2:35 in the afternoon"} \tab flexible, 12h \cr
   22 \tab \code{"Bh"} \tab \code{"2 in the afternoon"} \tab flexible, 12h \cr
   23 \tab \code{"hmsv"} \tab \code{"2:35:00 PM GMT+00:00"} \tab flexible, 12h \cr
   24 \tab \code{"hmv"} \tab \code{"2:35 PM GMT+00:00"} \tab flexible, 12h \cr
   25 \tab \code{"ms"} \tab \code{"35:00"} \tab flexible \cr
}


We can call \code{\link[=info_time_style]{info_time_style()}} in the console to view a similar table of
time styles with example output.
}

\section{Formatting with a \emph{CLDR} datetime pattern}{


We can use a \emph{CLDR} datetime pattern with the \code{format} argument to create
a highly customized and locale-aware output. This is a character string that
consists of two types of elements:
\itemize{
\item Pattern fields, which repeat a specific pattern character one or more
times. These fields are replaced with date and time data when formatting.
The character sets of \code{A}-\code{Z} and \code{a}-\code{z} are reserved for use as pattern
characters.
\item Literal text, which is output verbatim when formatting. This can include:
\itemize{
\item Any characters outside the reserved character sets, including
spaces and punctuation.
\item Any text between single vertical quotes (e.g., \code{'text'}).
\item Two adjacent single vertical quotes (''), which represent a literal
single quote, either inside or outside quoted text.
}
}

The number of pattern fields is quite sizable so let's first look at how some
\emph{CLDR} datetime patterns work. We'll use the datetime string
\code{"2018-07-04T22:05:09.2358(America/Vancouver)"} for all of the examples that
follow.
\itemize{
\item \code{"mm/dd/y"} -> \code{"05/04/2018"}
\item \code{"EEEE, MMMM d, y"} -> \code{"Wednesday, July 4, 2018"}
\item \code{"MMM d E"} -> \code{"Jul 4 Wed"}
\item \code{"HH:mm"} -> \code{"22:05"}
\item \code{"h:mm a"} -> \code{"10:05 PM"}
\item \code{"EEEE, MMMM d, y 'at' h:mm a"} -> \code{"Wednesday, July 4, 2018 at 10:05 PM"}
}

Here are the individual pattern fields:
\subsection{Year}{
\subsection{Calendar Year}{

This yields the calendar year, which is always numeric. In most cases the
length of the \code{"y"} field specifies the minimum number of digits to display,
zero-padded as necessary. More digits will be displayed if needed to show the
full year. There is an exception: \code{"yy"} gives use just the two low-order
digits of the year, zero-padded as necessary. For most use cases, \code{"y"} or
\code{"yy"} should be good enough.\tabular{ll}{
   Field Patterns \tab Output \cr
   \code{"y"} \tab \code{"2018"} \cr
   \code{"yy"} \tab \code{"18"} \cr
   \code{"yyy"} to \code{"yyyyyyyyy"} \tab \code{"2018"} to \code{"000002018"} \cr
}

}

\subsection{Year in the Week in Year Calendar}{

This is the year in 'Week of Year' based calendars in which the year
transition occurs on a week boundary. This may differ from calendar year
\code{"y"} near a year transition. This numeric year designation is used in
conjunction with pattern character \code{"w"} in the ISO year-week calendar as
defined by ISO 8601.\tabular{ll}{
   Field Patterns \tab Output \cr
   \code{"Y"} \tab \code{"2018"} \cr
   \code{"YY"} \tab \code{"18"} \cr
   \code{"YYY"} to \code{"YYYYYYYYY"} \tab \code{"2018"} to \code{"000002018"} \cr
}

}

}

\subsection{Quarter}{
\subsection{Quarter of the Year: formatting and standalone versions}{

The quarter names are identified numerically, starting at \code{1} and ending at
\code{4}. Quarter names may vary along two axes: the width and the context. The
context is either 'formatting' (taken as a default), which the form used
within a complete date format string, or, 'standalone', the form for date
elements used independently (such as in calendar headers). The standalone
form may be used in any other date format that shares the same form of the
name. Here, the formatting form for quarters of the year consists of some run
of \code{"Q"} values whereas the standalone form uses \code{"q"}.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"Q"}/\code{"q"} \tab \code{"3"} \tab Numeric, one digit \cr
   \code{"QQ"}/\code{"qq"} \tab \code{"03"} \tab Numeric, two digits (zero padded) \cr
   \code{"QQQ"}/\code{"qqq"} \tab \code{"Q3"} \tab Abbreviated \cr
   \code{"QQQQ"}/\code{"qqqq"} \tab \code{"3rd quarter"} \tab Wide \cr
   \code{"QQQQQ"}/\code{"qqqqq"} \tab \code{"3"} \tab Narrow \cr
}

}

}

\subsection{Month}{
\subsection{Month: formatting and standalone versions}{

The month names are identified numerically, starting at \code{1} and ending at
\code{12}. Month names may vary along two axes: the width and the context. The
context is either 'formatting' (taken as a default), which the form used
within a complete date format string, or, 'standalone', the form for date
elements used independently (such as in calendar headers). The standalone
form may be used in any other date format that shares the same form of the
name. Here, the formatting form for months consists of some run of \code{"M"}
values whereas the standalone form uses \code{"L"}.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"M"}/\code{"L"} \tab \code{"7"} \tab Numeric, minimum digits \cr
   \code{"MM"}/\code{"LL"} \tab \code{"07"} \tab Numeric, two digits (zero padded) \cr
   \code{"MMM"}/\code{"LLL"} \tab \code{"Jul"} \tab Abbreviated \cr
   \code{"MMMM"}/\code{"LLLL"} \tab \code{"July"} \tab Wide \cr
   \code{"MMMMM"}/\code{"LLLLL"} \tab \code{"J"} \tab Narrow \cr
}

}

}

\subsection{Week}{
\subsection{Week of Year}{

Values calculated for the week of year range from \code{1} to \code{53}. Week \code{1} for a
year is the first week that contains at least the specified minimum number of
days from that year. Weeks between week \code{1} of one year and week \code{1} of the
following year are numbered sequentially from \code{2} to \code{52} or \code{53} (if
needed).

There are two available field lengths. Both will display the week of year
value but the \code{"ww"} width will always show two digits (where weeks \code{1} to
\code{9} are zero padded).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"w"} \tab \code{"27"} \tab Minimum digits \cr
   \code{"ww"} \tab \code{"27"} \tab Two digits (zero padded) \cr
}

}

\subsection{Week of Month}{

The week of a month can range from \code{1} to \code{5}. The first day of every month
always begins at week \code{1} and with every transition into the beginning of a
week, the week of month value is incremented by \code{1}.\tabular{ll}{
   Field Pattern \tab Output \cr
   \code{"W"} \tab \code{"1"} \cr
}

}

}

\subsection{Day}{
\subsection{Day of Month}{

The day of month value is always numeric and there are two available field
length choices in its formatting. Both will display the day of month value
but the \code{"dd"} formatting will always show two digits (where days \code{1} to \code{9}
are zero padded).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"d"} \tab \code{"4"} \tab Minimum digits \cr
   \code{"dd"} \tab \code{"04"} \tab Two digits, zero padded \cr
}

}

\subsection{Day of Year}{

The day of year value ranges from \code{1} (January 1) to either \code{365} or \code{366}
(December 31), where the higher value of the range indicates that the year is
a leap year (29 days in February, instead of 28). The field length specifies
the minimum number of digits, with zero-padding as necessary.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"D"} \tab \code{"185"} \tab  \cr
   \code{"DD"} \tab \code{"185"} \tab Zero padded to minimum width of 2 \cr
   \code{"DDD"} \tab \code{"185"} \tab Zero padded to minimum width of 3 \cr
}

}

\subsection{Day of Week in Month}{

The day of week in month returns a numerical value indicating the number of
times a given weekday had occurred in the month (e.g., '2nd Monday in
March'). This conveniently resolves to predicable case structure where ranges
of day of the month values return predictable day of week in month values:
\itemize{
\item days \code{1} - \code{7} -> \code{1}
\item days \code{8} - \code{14} -> \code{2}
\item days \code{15} - \code{21} -> \code{3}
\item days \code{22} - \code{28} -> \code{4}
\item days \code{29} - \code{31} -> \code{5}
}\tabular{ll}{
   Field Pattern \tab Output \cr
   \code{"F"} \tab \code{"1"} \cr
}

}

\subsection{Modified Julian Date}{

The modified version of the Julian date is obtained by subtracting
2,400,000.5 days from the Julian date (the number of days since January 1,
4713 BC). This essentially results in the number of days since midnight
November 17, 1858. There is a half day offset (unlike the Julian date, the
modified Julian date is referenced to midnight instead of noon).\tabular{ll}{
   Field Patterns \tab Output \cr
   \code{"g"} to \code{"ggggggggg"} \tab \code{"58303"} -> \code{"000058303"} \cr
}

}

}

\subsection{Weekday}{
\subsection{Day of Week Name}{

The name of the day of week is offered in four different widths.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"E"}, \code{"EE"}, or \code{"EEE"} \tab \code{"Wed"} \tab Abbreviated \cr
   \code{"EEEE"} \tab \code{"Wednesday"} \tab Wide \cr
   \code{"EEEEE"} \tab \code{"W"} \tab Narrow \cr
   \code{"EEEEEE"} \tab \code{"We"} \tab Short \cr
}

}

}

\subsection{Periods}{
\subsection{AM/PM Period of Day}{

This denotes before noon and after noon time periods. May be upper or
lowercase depending on the locale and other options. The wide form may be
the same as the short form if the 'real' long form (e.g. 'ante meridiem') is
not customarily used. The narrow form must be unique, unlike some other
fields.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"a"}, \code{"aa"}, or \code{"aaa"} \tab \code{"PM"} \tab Abbreviated \cr
   \code{"aaaa"} \tab \code{"PM"} \tab Wide \cr
   \code{"aaaaa"} \tab \code{"p"} \tab Narrow \cr
}

}

\subsection{AM/PM Period of Day Plus Noon and Midnight}{

Provide AM and PM as well as phrases for exactly noon and midnight. May be
upper or lowercase depending on the locale and other options. If the locale
doesn't have the notion of a unique 'noon' (i.e., 12:00), then the PM form
may be substituted. A similar behavior can occur for 'midnight' (00:00) and
the AM form. The narrow form must be unique, unlike some other fields.

(a) \code{input_midnight}: \code{"2020-05-05T00:00:00"}
(b) \code{input_noon}: \code{"2020-05-05T12:00:00"}\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"b"}, \code{"bb"}, or \code{"bbb"} \tab (a) \code{"midnight"} \tab Abbreviated \cr
    \tab (b) \code{"noon"} \tab  \cr
   \code{"bbbb"} \tab (a) \code{"midnight"} \tab Wide \cr
    \tab (b) \code{"noon"} \tab  \cr
   \code{"bbbbb"} \tab (a) \code{"mi"} \tab Narrow \cr
    \tab (b) \code{"n"} \tab  \cr
}

}

\subsection{Flexible Day Periods}{

Flexible day periods denotes things like 'in the afternoon', 'in the
evening', etc., and the flexibility comes from a locale's language and
script. Each locale has an associated rule set that specifies when the day
periods start and end for that locale.

(a) \code{input_morning}: \code{"2020-05-05T00:08:30"}
(b) \code{input_afternoon}: \code{"2020-05-05T14:00:00"}\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"B"}, \code{"BB"}, or \code{"BBB"} \tab (a) \code{"in the morning"} \tab Abbreviated \cr
    \tab (b) \code{"in the afternoon"} \tab  \cr
   \code{"BBBB"} \tab (a) \code{"in the morning"} \tab Wide \cr
    \tab (b) \code{"in the afternoon"} \tab  \cr
   \code{"BBBBB"} \tab (a) \code{"in the morning"} \tab Narrow \cr
    \tab (b) \code{"in the afternoon"} \tab  \cr
}

}

}

\subsection{Hours, Minutes, and Seconds}{
\subsection{Hour 0-23}{

Hours from \code{0} to \code{23} are for a standard 24-hour clock cycle (midnight plus
1 minute is \code{00:01}) when using \code{"HH"} (which is the more common width that
indicates zero-padding to 2 digits).

Using \code{"2015-08-01T08:35:09"}:\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"H"} \tab \code{"8"} \tab Numeric, minimum digits \cr
   \code{"HH"} \tab \code{"08"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Hour 1-12}{

Hours from \code{1} to \code{12} are for a standard 12-hour clock cycle (midnight plus
1 minute is \code{12:01}) when using \code{"hh"} (which is the more common width that
indicates zero-padding to 2 digits).

Using \code{"2015-08-01T08:35:09"}:\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"h"} \tab \code{"8"} \tab Numeric, minimum digits \cr
   \code{"hh"} \tab \code{"08"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Hour 1-24}{

Using hours from \code{1} to \code{24} is a less common way to express a 24-hour clock
cycle (midnight plus 1 minute is \code{24:01}) when using \code{"kk"} (which is the
more common width that indicates zero-padding to 2 digits).

Using \code{"2015-08-01T08:35:09"}:\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"k"} \tab \code{"9"} \tab Numeric, minimum digits \cr
   \code{"kk"} \tab \code{"09"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Hour 0-11}{

Using hours from \code{0} to \code{11} is a less common way to express a 12-hour clock
cycle (midnight plus 1 minute is \code{00:01}) when using \code{"KK"} (which is the
more common width that indicates zero-padding to 2 digits).

Using \code{"2015-08-01T08:35:09"}:\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"K"} \tab \code{"7"} \tab Numeric, minimum digits \cr
   \code{"KK"} \tab \code{"07"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Minute}{

The minute of the hour which can be any number from \code{0} to \code{59}. Use \code{"m"} to
show the minimum number of digits, or \code{"mm"} to always show two digits
(zero-padding, if necessary).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"m"} \tab \code{"5"} \tab Numeric, minimum digits \cr
   \code{"mm"} \tab \code{"06"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Seconds}{

The second of the minute which can be any number from \code{0} to \code{59}. Use \code{"s"}
to show the minimum number of digits, or \code{"ss"} to always show two digits
(zero-padding, if necessary).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"s"} \tab \code{"9"} \tab Numeric, minimum digits \cr
   \code{"ss"} \tab \code{"09"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Fractional Second}{

The fractional second truncates (like other time fields) to the width
requested (i.e., count of letters). So using pattern \code{"SSSS"} will display
four digits past the decimal (which, incidentally, needs to be added manually
to the pattern).\tabular{ll}{
   Field Patterns \tab Output \cr
   \code{"S"} to \code{"SSSSSSSSS"} \tab \code{"2"} -> \code{"235000000"} \cr
}

}

\subsection{Milliseconds Elapsed in Day}{

There are 86,400,000 milliseconds in a day and the \code{"A"} pattern will provide
the whole number. The width can go up to nine digits with \code{"AAAAAAAAA"} and
these higher field widths will result in zero padding if necessary.

Using \code{"2011-07-27T00:07:19.7223"}:\tabular{ll}{
   Field Patterns \tab Output \cr
   \code{"A"} to \code{"AAAAAAAAA"} \tab \code{"439722"} -> \code{"000439722"} \cr
}

}

}

\subsection{Era}{
\subsection{The Era Designator}{

This provides the era name for the given date. The Gregorian calendar has two
eras: AD and BC. In the AD year numbering system, AD 1 is immediately
preceded by 1 BC, with nothing in between them (there was no year zero).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"G"}, \code{"GG"}, or \code{"GGG"} \tab \code{"AD"} \tab Abbreviated \cr
   \code{"GGGG"} \tab \code{"Anno Domini"} \tab Wide \cr
   \code{"GGGGG"} \tab \code{"A"} \tab Narrow \cr
}

}

}

\subsection{Time Zones}{
\subsection{TZ // Short and Long Specific non-Location Format}{

The short and long specific non-location formats for time zones are suggested
for displaying a time with a user friendly time zone name. Where the short
specific format is unavailable, it will fall back to the short localized GMT
format (\code{"O"}). Where the long specific format is unavailable, it will fall
back to the long localized GMT format (\code{"OOOO"}).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"z"}, \code{"zz"}, or \code{"zzz"} \tab \code{"PDT"} \tab Short Specific \cr
   \code{"zzzz"} \tab \code{"Pacific Daylight Time"} \tab Long Specific \cr
}

}

\subsection{TZ // Common UTC Offset Formats}{

The ISO8601 basic format with hours, minutes and optional seconds fields is
represented by \code{"Z"}, \code{"ZZ"}, or \code{"ZZZ"}. The format is equivalent to RFC 822
zone format (when the optional seconds field is absent). This is equivalent
to the \code{"xxxx"} specifier. The field pattern \code{"ZZZZ"} represents the long
localized GMT format. This is equivalent to the \code{"OOOO"} specifier. Finally,
\code{"ZZZZZ"} pattern yields the ISO8601 extended format with hours, minutes and
optional seconds fields. The ISO8601 UTC indicator \code{Z} is used when local
time offset is \code{0}. This is equivalent to the \code{"XXXXX"} specifier.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"Z"}, \code{"ZZ"}, or \code{"ZZZ"} \tab \code{"-0700"} \tab ISO 8601 basic format \cr
   \code{"ZZZZ"} \tab \code{"GMT-7:00"} \tab Long localized GMT format \cr
   \code{"ZZZZZ"} \tab \code{"-07:00"} \tab ISO 8601 extended format \cr
}

}

\subsection{TZ // Short and Long Localized GMT Formats}{

The localized GMT formats come in two widths \code{"O"} (which removes the minutes
field if it's \code{0}) and \code{"OOOO"} (which always contains the minutes field).
The use of the \code{GMT} indicator changes according to the locale.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"O"} \tab \code{"GMT-7"} \tab Short localized GMT format \cr
   \code{"OOOO"} \tab \code{"GMT-07:00"} \tab Long localized GMT format \cr
}

}

\subsection{TZ // Short and Long Generic non-Location Formats}{

The generic non-location formats are useful for displaying a recurring wall
time (e.g., events, meetings) or anywhere people do not want to be overly
specific. Where either of these is unavailable, there is a fallback to the
generic location format (\code{"VVVV"}), then the short localized GMT format as
the final fallback.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"v"} \tab \code{"PT"} \tab Short generic non-location format \cr
   \code{"vvvv"} \tab \code{"Pacific Time"} \tab Long generic non-location format \cr
}

}

\subsection{TZ // Short Time Zone IDs and Exemplar City Formats}{

These formats provide variations of the time zone ID and often include the
exemplar city. The widest of these formats, \code{"VVVV"}, is useful for
populating a choice list for time zones, because it supports 1-to-1 name/zone
ID mapping and is more uniform than other text formats.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"V"} \tab \code{"cavan"} \tab Short time zone ID \cr
   \code{"VV"} \tab \code{"America/Vancouver"} \tab Long time zone ID \cr
   \code{"VVV"} \tab \code{"Vancouver"} \tab The tz exemplar city \cr
   \code{"VVVV"} \tab \code{"Vancouver Time"} \tab Generic location format \cr
}

}

\subsection{TZ // ISO 8601 Formats with Z for +0000}{

The \code{"X"}-\code{"XXX"} field patterns represent valid ISO 8601 patterns for time
zone offsets in datetimes. The final two widths, \code{"XXXX"} and \code{"XXXXX"} allow
for optional seconds fields. The seconds field is \emph{not} supported by the ISO
8601 specification. For all of these, the ISO 8601 UTC indicator \code{Z} is used
when the local time offset is \code{0}.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"X"} \tab \code{"-07"} \tab ISO 8601 basic format (h, optional m) \cr
   \code{"XX"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m) \cr
   \code{"XXX"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m) \cr
   \code{"XXXX"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m, optional s) \cr
   \code{"XXXXX"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m, optional s) \cr
}

}

\subsection{TZ // ISO 8601 Formats (no use of Z for +0000)}{

The \code{"x"}-\code{"xxxxx"} field patterns represent valid ISO 8601 patterns for time
zone offsets in datetimes. They are similar to the \code{"X"}-\code{"XXXXX"} field
patterns except that the ISO 8601 UTC indicator \code{Z} \emph{will not} be used when
the local time offset is \code{0}.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"x"} \tab \code{"-07"} \tab ISO 8601 basic format (h, optional m) \cr
   \code{"xx"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m) \cr
   \code{"xxx"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m) \cr
   \code{"xxxx"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m, optional s) \cr
   \code{"xxxxx"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m, optional s) \cr
}

}

}
}

\section{Formatting with a \code{strptime} format code}{


Performing custom date/time formatting with the \code{format} argument can also
occur with a \code{strptime} format code. This works by constructing a string of
individual format codes representing formatted date and time elements. These
are all indicated with a leading \verb{\%}, literal characters are interpreted as
any characters not starting with a \verb{\%} character.

First off, let's look at a few format code combinations that work well
together as a \code{strptime} format. This will give us an intuition on how these
generally work. We'll use the datetime \code{"2015-06-08 23:05:37.48"} for all of
the examples that follow.
\itemize{
\item \code{"\%m/\%d/\%Y"} -> \code{"06/08/2015"}
\item \code{"\%A, \%B \%e, \%Y"} -> \code{"Monday, June 8, 2015"}
\item \code{"\%b \%e \%a"} -> \code{"Jun 8 Mon"}
\item \code{"\%H:\%M"} -> \code{"23:05"}
\item \code{"\%I:\%M \%p"} -> \code{"11:05 pm"}
\item \code{"\%A, \%B \%e, \%Y at \%I:\%M \%p"} -> \code{"Monday, June 8, 2015 at 11:05 pm"}
}

Here are the individual format codes for the date components:
\itemize{
\item \code{"\%a"} -> \code{"Mon"} (abbreviated day of week name)
\item \code{"\%A"} -> \code{"Monday"} (full day of week name)
\item \code{"\%w"} -> \code{"1"} (day of week number in \verb{0..6}; Sunday is \code{0})
\item \code{"\%u"} -> \code{"1"} (day of week number in \verb{1..7}; Monday is \code{1}, Sunday \code{7})
\item \code{"\%y"} -> \code{"15"} (abbreviated year, using the final two digits)
\item \code{"\%Y"} -> \code{"2015"} (full year)
\item \code{"\%b"} -> \code{"Jun"} (abbreviated month name)
\item \code{"\%B"} -> \code{"June"} (full month name)
\item \code{"\%m"} -> \code{"06"} (month number)
\item \code{"\%d"} -> \code{"08"} (day number, zero-padded)
\item \code{"\%e"} -> \code{"8"} (day number without zero padding)
\item \code{"\%j"} -> \code{"159"} (day of the year, always zero-padded)
\item \code{"\%W"} -> \code{"23"} (week number for the year, always zero-padded)
\item \code{"\%V"} -> \code{"24"} (week number for the year, following the ISO 8601
standard)
\item \code{"\%C"} -> \code{"20"} (the century number)
}

Here are the individual format codes for the time components:
\itemize{
\item \code{"\%H"} -> \code{"23"} (24h hour)
\item \code{"\%I"} -> \code{"11"} (12h hour)
\item \code{"\%M"} -> \code{"05"} (minute)
\item \code{"\%S"} -> \code{"37"} (second)
\item \code{"\%OS3"} -> \code{"37.480"} (seconds with decimals; \code{3} decimal places here)
\item \verb{\%p} -> \code{"pm"} (AM or PM indicator)
}

Here are some extra formats that you may find useful:
\itemize{
\item \code{"\%z"} -> \code{"+0000"} (signed time zone offset, here using UTC)
\item \code{"\%F"} -> \code{"2015-06-08"} (the date in the ISO 8601 date format)
\item \code{"\%\%"} -> \code{"\%"} (the literal "\verb{\%}" character, in case you need it)
}
}

\section{Adapting output to a specific \code{locale}}{


This formatting function can adapt outputs according to a provided \code{locale}
value. Examples include \code{"en"} for English (United States) and \code{"fr"} for
French (France). Note that a \code{locale} value provided here will override any
global locale setting performed in \code{\link[=gt]{gt()}}'s own \code{locale} argument (it is
settable there as a value received by all other functions that have a
\code{locale} argument). As a useful reference on which locales are supported, we
can call \code{\link[=info_locales]{info_locales()}} to view an info table.
}

\section{Examples}{


Use the \code{\link{exibble}} dataset to create a single-column \strong{gt} table (with only
the \code{datetime} column). With \code{fmt_datetime()} we'll format the \code{datetime}
column to have dates formatted with the \code{"month_day_year"} style and times
with the \code{"h_m_s_p"} 12-hour time style.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  dplyr::select(datetime) |>
  gt() |>
  fmt_datetime(
    date_style = "month_day_year",
    time_style = "h_m_s_p"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_datetime_1.png" alt="This image of a table was generated from the first code example in the `fmt_datetime()` help file." style="width:100\%;">
}}

Using the same input table, we can use \code{fmt_datetime()} with flexible date
and time styles. Two that work well together are \code{"MMMEd"} and \code{"Hms"}. These
date and time styles will, being flexible, create outputs that conform to the
locale value given to the \code{locale} argument. Let's use two calls of
\code{fmt_datetime()}: the first will format all rows in \code{datetime} to the Danish
locale (with \code{locale = "da"}) and the second call will target the first three
rows with the same formatting, but in the default locale (which is \code{"en"}).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  dplyr::select(datetime) |>
  gt() |>
  fmt_datetime(
    date_style = "MMMEd",
    time_style = "Hms",
    locale = "da"
  ) |>
  fmt_datetime(
    rows = 1:3,
    date_style = "MMMEd",
    time_style = "Hms"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_datetime_2.png" alt="This image of a table was generated from the second code example in the `fmt_datetime()` help file." style="width:100\%;">
}}

It's possible to use the \code{format} argument and write our own formatting
specification. Using the CLDR datetime pattern
\code{"EEEE, MMMM d, y 'at' h:mm a (zzzz)"} gives us datetime outputs with time
zone formatting. Let's provide a time zone ID (\code{"America/Vancouver"}) to the
\code{tz} argument.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  dplyr::select(datetime) |>
  gt() |>
  fmt_datetime(
    format = "EEEE, MMMM d, y 'at' h:mm a (zzzz)",
    tz = "America/Vancouver"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_datetime_3.png" alt="This image of a table was generated from the third code example in the `fmt_datetime()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-15
}

\section{Function Introduced}{

\code{v0.2.0.5} (March 31, 2020)
}

\seealso{
The vector-formatting version of this function:
\code{\link[=vec_fmt_datetime]{vec_fmt_datetime()}}.

Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_chem}()},
\code{\link{fmt_country}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_date}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_email}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_icon}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_tf}()},
\code{\link{fmt_time}()},
\code{\link{fmt_units}()},
\code{\link{fmt_url}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
