% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xcorr.R
\name{xcorr}
\alias{xcorr}
\title{Cross-correlation}
\usage{
xcorr(
  x,
  y = NULL,
  maxlag = if (is.matrix(x)) nrow(x) - 1 else max(length(x), length(y)) - 1,
  scale = c("none", "biased", "unbiased", "coeff")
)
}
\arguments{
\item{x}{Input, numeric or complex vector or matrix. Must not be missing.}

\item{y}{Input, numeric or complex vector data.  If \code{x} is a matrix (not
a vector), \code{y} must be omitted. \code{y} may be omitted if \code{x} is
a vector; in this case \code{xcorr} estimates the autocorrelation of
\code{x}.}

\item{maxlag}{Integer scalar. Maximum correlation lag. If omitted, the
default value is \code{N-1}, where \code{N} is the greater of the lengths
of \code{x} and \code{y} or, if \code{x} is a matrix, the number of rows in
\code{x}.}

\item{scale}{Character string. Specifies the type of scaling applied to the
 correlation vector (or matrix). matched to one of:
 \describe{
   \item{"none"}{return the unscaled correlation, R}
   \item{"biased"}{return the biased average, R / N}
   \item{"unbiased"}{return the unbiased average, R(k) / (N - |k|)}
   \item{"coeff"}{return the correlation coefficient, R / (rms(x) .
   rms(y))}, where \code{k} is the lag, and \code{N} is the length of
   \code{x}
 }
If omitted, the default value is \code{"none"}. If \code{y} is supplied but
does not have the same length as \code{x}, scale must be \code{"none"}.}
}
\value{
A list containing the following variables:
\describe{
  \item{R}{array of correlation estimates}
  \item{lags}{vector of correlation lags \code{[-maxlag:maxlag]}}
}
The array of correlation estimates has one of the following forms:
\enumerate{
  \item Cross-correlation estimate if X and Y are vectors.
  \item Autocorrelation estimate if is a vector and Y is omitted.
  \item If \code{x} is a matrix, \code{R} is a matrix containing the
  cross-correlation estimate of each column with every other column. Lag
  varies with the first index so that \code{R} has \code{2 * maxlag + 1} rows
  and \eqn{P^2} columns where \code{P} is the number of columns in \code{x}.
}
}
\description{
Estimate the cross-correlation between two sequences or the autocorrelation
of a single sequence
}
\details{
Estimate the cross correlation R_xy(k) of vector arguments \code{x} and
\code{y} or, if \code{y} is omitted, estimate autocorrelation R_xx(k) of
vector \code{x}, for a range of lags \code{k} specified by the argument
\code{maxlag}. If \code{x} is a matrix, each column of \code{x} is correlated
with itself and every other column.

The cross-correlation estimate between vectors \code{x} and \code{y} (of
length \code{N}) for lag \code{k} is given by
\if{latex}{
  \deqn{R_{xy}(k) = \sum_{i=1}^{N} x_{i+k} Conj(y_i)}
}
\if{html}{\preformatted{
            N
     Rxy = SUM x(i+k) . Conj(y(i))
           i=1
}}
where data not provided (for example \code{x[-1], y[N+1]}) is zero. Note the
definition of cross-correlation given above. To compute a cross-correlation
consistent with the field of statistics, see xcov.

The cross-correlation estimate is calculated by a "spectral" method
in which the FFT of the first vector is multiplied element-by-element
with the FFT of second vector.  The computational effort depends on
the length N of the vectors and is independent of the number of lags
requested.  If you only need a few lags, the "direct sum" method may
be faster.
}
\examples{
## Create a vector x and a vector y that is equal to x shifted by 5
## elements to the right. Compute and plot the estimated cross-correlation
## of x and y. The largest spike occurs at the lag value when the elements
## of x and y match exactly (-5).
n <- 0:15
x <- 0.84^n
y <- pracma::circshift(x, 5)
rl <- xcorr(x, y)
plot(rl$lag, rl$R, type="h")

## Compute and plot the estimated autocorrelation of a vector x.
## The largest spike occurs at zero lag, when x matches itself exactly.
n <- 0:15
x <- 0.84^n
rl <- xcorr(x)
plot(rl$lag, rl$R, type="h")

## Compute and plot the normalized cross-correlation of vectors
## x and y with unity peak, and specify a maximum lag of 10.
n <- 0:15
x <- 0.84^n
y <- pracma::circshift(x, 5)
rl <- xcorr(x, y, 10, 'coeff')
plot(rl$lag, rl$R, type="h")

}
\seealso{
\code{\link{xcov}}.
}
\author{
Paul Kienzle, \email{pkienzle@users.sf.net},\cr
  Asbjorn Sabo, \email{asbjorn.sabo@broadpark.no},\cr
  Peter Lanspeary. \email{peter.lanspeary@adelaide.edu.au}.\cr
   Conversion to R by Geert van Boxtel, \email{G.J.M.vanBoxtel@gmail.com}.
}
