% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GIRFandGFEVD.R, R/plotMethods.R,
%   R/printMethods.R
\name{GFEVD}
\alias{GFEVD}
\alias{plot.gfevd}
\alias{print.gfevd}
\title{Estimate generalized forecast error variance decomposition for structural
 (and reduced form) GMVAR, StMVAR, and G-StMVAR models.}
\usage{
GFEVD(
  gsmvar,
  shock_size = 1,
  N = 30,
  initval_type = c("data", "random", "fixed"),
  R1 = 250,
  R2 = 250,
  init_regimes = NULL,
  init_values = NULL,
  which_cumulative = numeric(0),
  include_mixweights = FALSE,
  ncores = 2,
  seeds = NULL
)

\method{plot}{gfevd}(x, ...)

\method{print}{gfevd}(x, ..., digits = 2, N_to_print)
}
\arguments{
\item{gsmvar}{an object of class \code{'gsmvar'}, typically created with \code{fitGSMVAR} or \code{GSMVAR}.}

\item{shock_size}{What shocks size should be used for all shocks? By the definition of the SGMVAR,
SStMVAR, and SG-StMVAR model, the conditional covariance matrix of the structural shock is identity matrix.}

\item{N}{a positive integer specifying the horizon how far ahead should the GFEVD be calculated.}

\item{initval_type}{What type initial values are used for estimating the GIRFs that the GFEVD is based on?
\describe{
  \item{\code{"data"}:}{Estimate the GIRF for all the possible length \eqn{p} histories in the data.}
  \item{\code{"random"}:}{Estimate the GIRF for several random initial values generated from the stationary
     distribution of the process or from the stationary distribution of specific regime(s) chosen with the
     argument \code{init_regimes}. The number of initial values is set with the argument \code{R2}.}
  \item{\code{"fixed"}:}{Estimate the GIRF for a fixed initial value only, which is specified with the argument
     \code{init_values}.}
}}

\item{R1}{the number of repetitions used to estimate GIRF for each initial
value.}

\item{R2}{the number of initial values to be drawn if \code{initval_type="random"}.}

\item{init_regimes}{a numeric vector of length at most \eqn{M} and elements
in \eqn{1,...,M} specifying the regimes from which the initial values
should be generated from. The initial values will be generated from a
mixture distribution with the mixture components being the stationary
distributions of the specific regimes and the (proportional) mixing weights
given by the mixing weight parameters of those regimes. Note that if
\code{init_regimes=1:M}, the initial values are generated from the
stationary distribution of the process and if \code{init_regimes=m}, the
initial value are generated from the stationary distribution of the
\eqn{m}th regime. Ignored if the argument \code{init_values} is specified.}

\item{init_values}{a size \eqn{(pxd)} matrix specifying the initial values, where d is the number
of time series in the system. The \strong{last} row will be used as initial values for the first lag,
the second last row for second lag etc. If not specified, initial values will be drawn according to
mixture distribution specifed by the argument \code{init_regimes}.}

\item{which_cumulative}{a numeric vector with values in \eqn{1,...,d}
(\code{d=ncol(data)}) specifying which the variables for which the impulse
responses should be cumulative. Default is none.}

\item{include_mixweights}{should the GFEVD be estimated for the mixing weights as well? Note that this is
ignored if \code{M=1} and if \code{M=2} the GFEVD will be the same for both regime's mixing weights.}

\item{ncores}{the number CPU cores to be used in parallel computing. Only
single core computing is supported if an initial value is specified (and
the GIRF won't thus be estimated multiple times).}

\item{seeds}{a numeric vector containing the random number generator seed for estimation
of each GIRF. Should have the length...
\itemize{
  \item ...\code{nrow(data) - p + 1} if \code{initval_type="data"}.
  \item ...\code{R2} if \code{initval_type="random"}.
  \item ...\code{1} if \code{initval_type="fixed."}.
}
Set to \code{NULL} for not initializing the seed. Exists for creating reproducible results.}

\item{x}{object of class \code{'gfevd'} generated by the function \code{GFEVD}.}

\item{...}{currently not used.}

\item{digits}{the number of decimals to print}

\item{N_to_print}{an integer specifying the horizon how far to print the estimates.
The default is that all the values are printed.}
}
\value{
Returns and object of class 'gfevd' containing the GFEVD for all the variables and if
  \code{include_mixweights=TRUE} also to the mixing weights. Note that the decomposition does not
  exist at horizon zero for mixing weights because the related GIRFs are always zero at impact.
}
\description{
\code{GFEVD} estimates generalized forecast error variance decomposition for structural
 (and reduced form) GMVAR, StMVAR, and G-StMVAR models.
}
\details{
The model DOES NOT need to be structural in order for this function to be
  applicable. When an identified structural GMVAR, StMVAR, or G-StMVAR model is
  provided in the argument \code{gsmvar}, the identification imposed by the model
  is used. When a reduced form model is provided in the argument \code{gsmvar},
  lower triangular Cholesky identification is used to identify the shocks.

  The GFEVD is a forecast error variance decomposition calculated with the generalized impulse response function (GIRF).
  Lanne and Nyberg (2016) for details. Note, however, that the related GIRFs are calculated using the algorithm given in
  Virolainen (2022).
}
\section{Functions}{
\itemize{
\item \code{plot(gfevd)}: plot method

\item \code{print(gfevd)}: print method

}}
\examples{
 \donttest{
 # These are long-running examples that use parallel computing.
 # It takes approximately 30 seconds to run all the below examples.

 ## StMVAR(1, 2), d=2 model identified recursively by lower-triangular
 ## Cholesky decomposition (i.e., reduced form model is specified):
 params12t <- c(0.55, 0.11, 0.34, 0.05, -0.01, 0.72, 0.58, 0.01, 0.06, 0.17,
   0.25, 0.34, 0.05, -0.01, 0.72, 0.50, -0.01, 0.20, 0.60, 3.00, 12.00)
 mod12t <- GSMVAR(gdpdef, p=1, M=2, params=params12t, model="StMVAR")

 # Estimating the GFEVD using all possible histories in the data as the
 # initial values:
 gfevd0 <- GFEVD(mod12t, N=24, R1=10, initval_type="data")
 gfevd0
 plot(gfevd0)
 ## NOTE: Use larger R1 is empirical applications! Small R1 is used
 ## here only to fasten the execution time of the examples.

 ## Structural GMVAR(2, 2), d=2 model identified with sign-constraints:
 params22s <- c(0.36, 0.121, 0.484, 0.072, 0.223, 0.059, -0.151, 0.395,
  0.406, -0.005, 0.083, 0.299, 0.218, 0.02, -0.119, 0.722, 0.093, 0.032,
  0.044, 0.191, 0.057, 0.172, -0.46, 0.016, 3.518, 5.154, 0.58)
 W_22 <- matrix(c(1, 1, -1, 1), nrow=2, byrow=FALSE)
 mod22s <- GSMVAR(gdpdef, p=2, M=2, params=params22s,
  structural_pars=list(W=W_22))
 mod22s
 # Alternatively, use:
 #fit22s <- fitGSMVAR(gdpdef, p=2, M=2, structural_pars=list(W=W_22),
 #                   ncalls=20, seeds=1:20)
 # To obtain an estimated version of the same model.

 # Estimating the GFEVD using all possible histories in the data as the
 # initial values:
 gfevd1 <- GFEVD(mod22s, N=24, R1=10, initval_type="data")
 gfevd1
 plot(gfevd1)

 # Estimate GFEVD with the initial values generated from the stationary
 # distribution of the process:
 gfevd2 <- GFEVD(mod22s, N=24, R1=10, R2=100, initval_type="random")
 gfevd2
 plot(gfevd2)

 # Estimate GFEVD with fixed hand specified initial values. We use the
 # unconditional mean of the process:
 myvals <- rbind(mod22s$uncond_moments$uncond_mean,
                 mod22s$uncond_moments$uncond_mean)
 gfevd3 <- GFEVD(mod22s, N=36, R1=50, initval_type="fixed",
  init_values=myvals, include_mixweights=TRUE)
 gfevd3
 plot(gfevd3)
 }
}
\references{
\itemize{
   \item Lanne M. and Nyberg H. 2016. Generalized Forecast Error Variance Decomposition for Linear
     and Nonlineae Multivariate Models. \emph{Oxford Bulletin of Economics and Statistics}, \strong{78}, 4, 595-603.
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2016. Gaussian mixture vector autoregression.
         \emph{Journal of Econometrics}, \strong{192}, 485-498.
   \item Virolainen S. 2022. Structural Gaussian mixture vector autoregressive model with application to the asymmetric
     effects of monetary policy shocks. Unpublished working paper, available as arXiv:2007.04713.
   \item Virolainen S. 2022. Gaussian and Student's t mixture vector autoregressive model with application to the
     asymmetric effects of monetary policy shocks in the Euro area. Unpublished working paper, available as arXiv:2109.13648.
 }
}
\seealso{
\code{\link{GIRF}}, \code{\link{linear_IRF}}, \code{\link{fitGSMVAR}}, \code{\link{GSMVAR}},
 \code{\link{gsmvar_to_sgsmvar}}, \code{\link{reorder_W_columns}}, \code{\link{swap_W_signs}}, \code{\link{simulate.gsmvar}}
}
