% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glms.R
\name{stepCriterion.lm}
\alias{stepCriterion.lm}
\title{Variable Selection in Normal Linear Models}
\usage{
\method{stepCriterion}{lm}(
  model,
  criterion = c("bic", "aic", "adjr2", "prdr2", "cp", "p-value"),
  direction = c("forward", "backward"),
  levels = c(0.05, 0.05),
  trace = TRUE,
  scope,
  ...
)
}
\arguments{
\item{model}{an object of the class \emph{lm}.}

\item{criterion}{an (optional) character string indicating the criterion which should be used to compare the candidate models. The available options are: AIC ("aic"), BIC ("bic"), adjusted R-squared ("adjr2"), predicted R-squared ("prdr2"), Mallows' CP ("cp") and \emph{p}-value of the F test ("p-value"). As default, \code{criterion} is set to "bic".}

\item{direction}{an (optional) character string indicating the type of procedure which should be used. The available options are: hybrid backward stepwise ("backward") and hybrid forward stepwise ("forward"). As default, \code{direction} is set to "forward".}

\item{levels}{an (optional) two-dimensional vector of values in the interval \eqn{(0,1)} indicating the levels at which the variables should in and out from the model. This is only appropiate if \code{criterion}="p-value". As default, \code{levels} is set to \code{c(0.05,0.05)}.}

\item{trace}{an (optional) logical switch indicating if should the stepwise reports be printed. As default, \code{trace} is set to TRUE.}

\item{scope}{an (optional) list containing components \code{lower} and \code{upper}, both formula-type objects, indicating the range of models which should be examined in the stepwise search. As default, \code{lower} is a model with no predictors and \code{upper} is the linear predictor of the model in \code{model}.}

\item{...}{further arguments passed to or from other methods. For example, \code{k}, that is, the magnitude of the penalty in the AIC/QICu, which by default is set to 2.}
}
\value{
a list list with components including
\tabular{ll}{
\code{initial} \tab  a character string indicating the linear predictor of the "initial model",\cr
\tab \cr
\code{direction} \tab  a character string indicating the type of procedure which was used,\cr
\tab \cr
\code{criterion} \tab a character string indicating the criterion used to compare the candidate models,\cr
\tab \cr
\code{final} \tab a character string indicating the linear predictor of the "final model",\cr
}
}
\description{
Performs variable selection in normal linear models using a hybrid versions of forward stepwise and backward stepwise.
}
\details{
The "hybrid forward stepwise" algorithm starts with the
simplest model (which may be chosen at the argument \code{scope}, and
As default, is a model whose parameters in the linear predictor,
except the intercept, if any, are set to 0), and then the candidate
models are built by hierarchically including effects in the linear
predictor, whose "relevance" and/or "importance" in the model fit is
assessed by comparing nested models (that is, by comparing the models
with and without the added effect) using a criterion previously
specified. If an effect is added to the equation, this strategy may
also remove any effect which, according to the previously specified
criteria, no longer provides an improvement in the model fit. That
process continues until no more effects are included or excluded. The
"hybrid backward stepwise" algorithm works similarly.
}
\examples{
###### Example 1: New York air quality measurements
fit1 <- lm(log(Ozone) ~ Solar.R + Temp + Wind, data=airquality)
scope=list(lower=~1, upper=~Solar.R*Temp*Wind)
stepCriterion(fit1, direction="forward", criterion="adjr2", scope=scope)
stepCriterion(fit1, direction="forward", criterion="bic", scope=scope)
stepCriterion(fit1, direction="forward", criterion="p-value", scope=scope)

###### Example 2: Fuel consumption of automobiles
fit2 <- lm(mpg ~ log(hp) + log(wt) + qsec, data=mtcars)
scope=list(lower=~1, upper=~log(hp)*log(wt)*qsec)
stepCriterion(fit2, direction="backward", criterion="bic", scope=scope)
stepCriterion(fit2, direction="forward", criterion="cp", scope=scope)
stepCriterion(fit2, direction="backward", criterion="prdr2", scope=scope)

###### Example 3: Credit card balance
Credit <- ISLR::Credit
fit3 <- lm(Balance ~ Cards + Age + Rating + Income + Student + Limit, data=Credit)
stepCriterion(fit3, direction="forward", criterion="prdr2")
stepCriterion(fit3, direction="forward", criterion="cp")
stepCriterion(fit3, direction="forward", criterion="p-value")

}
\references{
James G., Witten D., Hastie T., Tibshirani R. (2013, page 210) An Introduction to Statistical Learning with Applications in R, Springer, New York.
}
\seealso{
\link{stepCriterion.glm}, \link{stepCriterion.overglm}, \link{stepCriterion.glmgee}

\link{stepCriterion.glm}, \link{stepCriterion.overglm}, \link{stepCriterion.glmgee}
}
