% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{hskw}
\alias{hskw}
\title{Hessian Matrix of the Negative Log-Likelihood for the Kw Distribution}
\usage{
hskw(par, data)
}
\arguments{
\item{par}{A numeric vector of length 2 containing the distribution parameters
in the order: \code{alpha} (\eqn{\alpha > 0}), \code{beta} (\eqn{\beta > 0}).}

\item{data}{A numeric vector of observations. All values must be strictly
between 0 and 1 (exclusive).}
}
\value{
Returns a 2x2 numeric matrix representing the Hessian matrix of the
negative log-likelihood function, \eqn{-\partial^2 \ell / (\partial \theta_i \partial \theta_j)},
where \eqn{\theta = (\alpha, \beta)}.
Returns a 2x2 matrix populated with \code{NaN} if any parameter values are
invalid according to their constraints, or if any value in \code{data} is
not in the interval (0, 1).
}
\description{
Computes the analytic 2x2 Hessian matrix (matrix of second partial derivatives)
of the negative log-likelihood function for the two-parameter Kumaraswamy (Kw)
distribution with parameters \code{alpha} (\eqn{\alpha}) and \code{beta}
(\eqn{\beta}). The Hessian is useful for estimating standard errors and in
optimization algorithms.
}
\details{
This function calculates the analytic second partial derivatives of the
negative log-likelihood function (\eqn{-\ell(\theta|\mathbf{x})}). The components
are the negative of the second derivatives of the log-likelihood \eqn{\ell}
(derived from the PDF in \code{\link{dkw}}).

Let \eqn{v_i = 1 - x_i^{\alpha}}. The second derivatives of the positive log-likelihood (\eqn{\ell}) are:
\deqn{
\frac{\partial^2 \ell}{\partial \alpha^2} = -\frac{n}{\alpha^2} -
(\beta-1)\sum_{i=1}^{n}\frac{x_i^{\alpha}(\ln(x_i))^2}{v_i^2}
}
\deqn{
\frac{\partial^2 \ell}{\partial \alpha \partial \beta} = -
\sum_{i=1}^{n}\frac{x_i^{\alpha}\ln(x_i)}{v_i}
}
\deqn{
\frac{\partial^2 \ell}{\partial \beta^2} = -\frac{n}{\beta^2}
}
The function returns the Hessian matrix containing the negative of these values.

Key properties of the returned matrix:
\itemize{
\item Dimensions: 2x2.
\item Symmetry: The matrix is symmetric.
\item Ordering: Rows and columns correspond to the parameters in the order
\eqn{\alpha, \beta}.
\item Content: Analytic second derivatives of the \emph{negative} log-likelihood.
}
This corresponds to the relevant 2x2 submatrix of the 5x5 GKw Hessian (\code{\link{hsgkw}})
evaluated at \eqn{\gamma=1, \delta=0, \lambda=1}.
}
\examples{
\donttest{
# Assuming existence of rkw, llkw, grkw, hskw functions for Kw

# Generate sample data
set.seed(123)
true_par_kw <- c(alpha = 2, beta = 3)
sample_data_kw <- rkw(100, alpha = true_par_kw[1], beta = true_par_kw[2])
hist(sample_data_kw, breaks = 20, main = "Kw(2, 3) Sample")

# --- Find MLE estimates ---
start_par_kw <- c(1.5, 2.5)
mle_result_kw <- stats::optim(par = start_par_kw,
                              fn = llkw,
                              gr = if (exists("grkw")) grkw else NULL,
                              method = "L-BFGS-B",
                              lower = c(1e-6, 1e-6),
                              hessian = TRUE, # Ask optim for numerical Hessian
                              data = sample_data_kw)

# --- Compare analytical Hessian to numerical Hessian ---
if (mle_result_kw$convergence == 0 &&
    requireNamespace("numDeriv", quietly = TRUE) &&
    exists("hskw")) {

  mle_par_kw <- mle_result_kw$par
  cat("\nComparing Hessians for Kw at MLE estimates:\n")

  # Numerical Hessian of llkw
  num_hess_kw <- numDeriv::hessian(func = llkw, x = mle_par_kw, data = sample_data_kw)

  # Analytical Hessian from hskw
  ana_hess_kw <- hskw(par = mle_par_kw, data = sample_data_kw)

  cat("Numerical Hessian (Kw):\n")
  print(round(num_hess_kw, 4))
  cat("Analytical Hessian (Kw):\n")
  print(round(ana_hess_kw, 4))

  # Check differences
  cat("Max absolute difference between Kw Hessians:\n")
  print(max(abs(num_hess_kw - ana_hess_kw)))

  # Optional: Use analytical Hessian for Standard Errors
  # tryCatch({
  #   cov_matrix_kw <- solve(ana_hess_kw) # ana_hess_kw is already Hessian of negative LL
  #   std_errors_kw <- sqrt(diag(cov_matrix_kw))
  #   cat("Std. Errors from Analytical Kw Hessian:\n")
  #   print(std_errors_kw)
  # }, error = function(e) {
  #   warning("Could not invert analytical Kw Hessian: ", e$message)
  # })

} else {
  cat("\nSkipping Kw Hessian comparison.\n")
  cat("Requires convergence, 'numDeriv' package, and function 'hskw'.\n")
}

}

}
\references{
Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.

Jones, M. C. (2009). Kumaraswamy's distribution: A beta-type distribution
with some tractability advantages. \emph{Statistical Methodology}, \emph{6}(1), 70-81.

(Note: Specific Hessian formulas might be derived or sourced from additional references).
}
\seealso{
\code{\link{hsgkw}} (parent distribution Hessian),
\code{\link{llkw}} (negative log-likelihood for Kw),
\code{grkw} (gradient for Kw, if available),
\code{\link{dkw}} (density for Kw),
\code{\link[stats]{optim}},
\code{\link[numDeriv]{hessian}} (for numerical Hessian comparison).
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{hessian}
\keyword{kumaraswamy}
\keyword{likelihood}
\keyword{optimize}
