#' @name motif_location
#' @author Xiang LI <lixiang117423@@foxmail.com>
#'
#' @title Extract and Visualize Motif Information from MEME Software
#' @description
#' \code{getMotifFromXML} Visualize motif location in a specificial sequences..
#'
#' @param data A data frame file from getMotifFromXML function.
#'
#' @importFrom magrittr %>%
#' @importFrom dplyr select mutate
#' @importFrom ggplot2 ggplot aes geom_segment geom_rect scale_x_continuous labs theme_classic
#'
#' @examples
#' filepath <- system.file("examples", "meme.xml", package="ggmotif")
#' motif_extract <- getMotifFromXML(XMLfile = filepath)
#' motif_plot <- motif_location(data = motif_extract)
#'
#' @export
#'
#' @return Return a datafram
utils::globalVariables(c('seq.id','position','width','input.seq.id','motif_id',
                         'start.position','end.position','start','end','y','Genes',
                         'Motif','x.min','x.max','y.min','y.max'))

motif_location <- function(data) {

  my.gene <- data %>%
    dplyr::select(
      'input.seq.id', 'length', 'motif_id',
      'start.position', 'end.position'
    ) %>%
    dplyr::rename(
      Genes = input.seq.id, Motif = motif_id,
      start = start.position, end = end.position
    )

  my.gene.list <- data.frame(
    Genes = unique(my.gene$Genes),
    y = 1:length(unique(my.gene$Genes))
  )

  my.gene.final <- merge(my.gene, my.gene.list, by = "Genes") %>%
    dplyr::mutate(length = as.numeric(length)) %>%
    dplyr::mutate(x = max(length)) %>%
    dplyr::mutate(
      x.min = start,
      x.max = end,
      y.min = y - 0.4,
      y.max = y + 0.4,
      group = paste0(Genes, Motif)
    ) %>%
    dplyr::arrange(y) %>%
    dplyr::mutate(Genes = factor(Genes, levels = unique(Genes))) %>%
    dplyr::arrange(length)

  ggplot(my.gene.final) +
    geom_segment(aes(
      x = 0, xend = length,
      y = Genes, yend = Genes
    ),
    size = 0.8
    ) +
    geom_rect(aes(
      xmin = x.min, xmax = x.max,
      ymin = y.min, ymax = y.max,
      fill = Motif
    )) +
    scale_x_continuous(expand = c(0, 0), breaks = seq(0, max(my.gene.final$length), 100)) +
    labs(x = "", y = "Name") +
    theme_classic() -> p

  return(p)
}
